<?php

/*
* Plugin Name:       DaHannes Connector
* Plugin URI:        https://connector.dahannes.com
* Description:       Verbindet diese Website mit my.dahannes.com
* Version: 2.7.0
* Requires at least: 5.2
* Requires PHP:      8.1
* Author:            Skoric IT
* Author URI:        https://skoric.at
* License:           GPL v2 or later
* License URI:       https://www.gnu.org/licenses/gpl-2.0.html
* Text Domain:       dahannes-connector
* Domain Path:       /languages
*/

namespace DaHannesConnector;

const PLUGIN_VERSION = '2.7.0';  // Match this with the version in the plugin header

// Load plugin textdomain
add_action('init', function () {
    load_plugin_textdomain(
        'dahannes-connector',
        false,
        dirname(plugin_basename(__FILE__)) . '/languages'
    );
});

use DaHannesConnector\includes\AdminSettingsPage;
use DaHannesConnector\includes\CronScheduler;
use DaHannesConnector\includes\DahannesConnectorUtils;
use DaHannesConnector\includes\DahannesConfig;
use DaHannesConnector\includes\DahannesCronManager;
use DaHannesConnector\includes\DaHannes_Plugin_Update_Checker;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

// Load scoped dependencies autoloader
require_once __DIR__ . '/deps/scoper-autoload.php';

// Note: Sentry initialization is now handled by DahannesLogger class
// based on DAHANNES_SENTRY_DSN constant configuration

// Declare HPOS compatibility
add_action('before_woocommerce_init', function() {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

// Check for minimum PHP version
if (version_compare(PHP_VERSION, '8.1', '<')) {
    add_action('admin_notices', function () {
        ?>
        <div class="notice notice-error">
            <p><?php _e('DaHannes Connector requires PHP 8.1 or higher!', 'dahannes-connector'); ?></p>
        </div>
        <?php
    });
    return;
}

// Check for minimum WordPress version
if (version_compare($GLOBALS['wp_version'], '5.2', '<')) {
    add_action('admin_notices', function () {
        ?>
        <div class="notice notice-error">
            <p><?php _e('DaHannes Connector requires WordPress 5.2 or higher!', 'dahannes-connector'); ?></p>
        </div>
        <?php
    });
    return;
}

register_activation_hook(__FILE__, function () {
    // Check for WooCommerce
    if (!class_exists('WooCommerce')) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(__('Sorry, but this plugin requires WooCommerce to be installed and active.', 'dahannes-connector'));
    }

    try {
        // Initialize plugin options with default values (delete and recreate to ensure proper type)
        delete_option('dahannes_info');
        add_option('dahannes_info', []);
        if (false === get_option('dahannes_api_success')) {
            add_option('dahannes_api_success', false);
        }
        if (false === get_option('dahannes_last_api_call_date')) {
            add_option('dahannes_last_api_call_date', 0);
        }

        // Load required files for activation
        require_once 'includes/dahannes-config.php';
        require_once 'includes/dahannes-cron-manager.php';
        require_once 'includes/dahannes-migration-manager.php';
        require_once 'includes/dahannes-logger.php';
        require_once 'includes/dahannes-connector-utils.php';

        // Migrate existing API credentials to encrypted storage
        DahannesConfig::migrate_existing_credentials();
        
        // Run plugin migrations
        includes\DahannesMigrationManager::maybe_run_migrations();

        // Schedule cron job
        $cron_manager = new DahannesCronManager();
        $cron_manager->schedule_sync();
        
    } catch (Exception $e) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(__('Plugin activation failed: ', 'dahannes-connector') . $e->getMessage());
    }
});

register_deactivation_hook(__FILE__, function () {
    // Remove scheduled cron events
    wp_clear_scheduled_hook('dahannes_sync_orders');
    // Also clear legacy hook if it exists
    wp_clear_scheduled_hook('dh_sync_to_server');
});

class DahannesConnector
{
    private static string $_icon_url = '';

    public static function init(): void
    {
        // add capabilities
        add_action('admin_init', function () {
            $admin = get_role('administrator');
            $admin->add_cap('manage_dahannes_connector');
            $shop_manager = get_role('shop_manager');
            $shop_manager->add_cap('manage_dahannes_connector');
        });


        // Add menu item
        DahannesConnectorUtils::register_shortcodes();
        DahannesConnectorUtils::add_dahannes_order_metabox();
        AdminSettingsPage::init();
        // DISABLED: CronScheduler::init(); // Disabled to prevent duplicate order submissions - uses DahannesCronManager instead


        // Get shop type for validation rules
        $dahannes_info = get_option('dahannes_info', []);
        $shop_type = $dahannes_info['app_type'] ?? '';

        // Register additional checkout field for WooCommerce 8.9+ Block Checkout compatibility
        DahannesConnectorUtils::register_additional_checkout_field();
        
        // Keep classic checkout support for backward compatibility
        add_action('woocommerce_after_order_notes', [DahannesConnectorUtils::class, 'add_dh_checkout_field']);
        add_action('woocommerce_checkout_update_order_meta', [DahannesConnectorUtils::class, 'save_dh_checkout_field']);
        
        // Handle block checkout field saving (WooCommerce 8.9+)
        add_action('woocommerce_store_api_checkout_update_order_from_request', [DahannesConnectorUtils::class, 'save_block_checkout_field'], 10, 2);

        // Add checkout process validation
        add_action('woocommerce_checkout_process', function () use ($shop_type) {
            if (!isset($_POST['dh_checkout_field'])) {
                if ($shop_type !== 'partner') {
                    wc_add_notice(__('Bitte geben Sie einen Dahannes Mitgliedscode ein.', 'dahannes-connector'), 'error');
                }
                return;
            }

            $code = trim($_POST['dh_checkout_field']);
            if (empty($code)) {
                if ($shop_type !== 'partner') {
                    wc_add_notice(__('Bitte geben Sie einen Dahannes Mitgliedscode ein.', 'dahannes-connector'), 'error');
                }
                return;
            }

            // Validate format if code is provided
            if (strtolower($code) !== 'x' && !preg_match('/^[A-Za-z0-9]{5,7}$/', $code)) {
                $message = $shop_type === 'partner'
                    ? __('Der Mitgliedscode muss 5-7 Zeichen lang sein.', 'dahannes-connector')
                    : __('Der Mitgliedscode muss 5-7 Zeichen lang sein oder ein „x" sein.', 'dahannes-connector');
                wc_add_notice($message, 'error');
            }
        });
        
        // HPOS and legacy compatible hooks for order handling
        add_action('save_post_shop_order', [DahannesConnectorUtils::class, 'hook_save_post_shop_order']); // Legacy post-based orders
        add_action('woocommerce_new_order', [DahannesConnectorUtils::class, 'hook_new_order']); // HPOS new orders
        add_action('woocommerce_update_order', [DahannesConnectorUtils::class, 'hook_update_order']); // HPOS order updates
        // Ensure resubmission when status changes via gateway/automation (not just admin save)
        add_action('woocommerce_order_status_changed', [DahannesConnectorUtils::class, 'hook_order_status_changed'], 10, 4);

        // Register WooCommerce product tab and fields
        DahannesConnectorUtils::register_woocommerce_hooks();

        // For backward compatibility, keep the original hook
        add_action('woocommerce_product_options_pricing', [DahannesConnectorUtils::class, 'dahannes_add_custom_fields']);

        // Add hook to save custom product fields
        add_action('woocommerce_process_product_meta', [DahannesConnectorUtils::class, 'dahannes_save_custom_fields']);

    }

    /**
     * Get the base64 encoded menu icon with caching
     *
     * @return string Base64 encoded SVG icon
     */
    public static function get_menu_icon(): string
    {
        if (!self::$_icon_url) {
            self::$_icon_url = 'data:image/svg+xml;base64,' . base64_encode(file_get_contents(__DIR__ . '/assets/images/dahannes-icon.svg'));
        }
        return self::$_icon_url;
    }

}

// Hook into 'plugins_loaded' to ensure all plugins are loaded before checking dependencies
add_action('plugins_loaded', function () {
    // Check if WooCommerce is active
    if (!class_exists('WooCommerce')) {
        add_action('admin_notices', function () {
            ?>
            <div class="notice notice-error">
                <p><?php _e('DaHannes Connector requires WooCommerce to be installed and activated!', 'dahannes-connector'); ?></p>
            </div>
            <?php
        });
        return; // Stop loading the plugin
    }
    
    // Load plugin includes after WooCommerce check passes
    require_once 'includes/api-client.php';
    require_once 'includes/admin-settings-page.php';
    require_once 'includes/dahannes-connector-utils.php';
    require_once 'includes/dahannes-logger.php';
    require_once 'includes/dahannes-order-manager.php';
    require_once 'includes/dahannes-admin-ui.php';
    require_once 'includes/dahannes-config.php';
    require_once 'includes/dahannes-cron-manager.php';
    require_once 'includes/dahannes-migration-manager.php';
    require_once 'includes/cron-scheduler.php';
    require_once 'includes/update-checker.php';
    
    // Load WP-CLI commands if WP-CLI is available
    if (defined('WP_CLI') && WP_CLI) {
        require_once 'includes/wp-cli-commands.php';
    }
    
    // Run migrations on every plugin load (to handle updates)
    includes\DahannesMigrationManager::maybe_run_migrations();
    
    // Initialize plugin functionality
    DahannesConnector::init();
    new includes\DahannesAdminUI();
    new includes\DahannesCronManager();
    
    // Initialize update checker
    new DaHannes_Plugin_Update_Checker(
        'dahannes-connector/dahannes-connector.php',
        PLUGIN_VERSION,
        'https://connector.dahannes.com/updater/info.json'
    );
    
    // Hook for saving DH value to cookie and user meta
    add_action('init', [DahannesConnectorUtils::class, 'save_dh_value_to_cookie_and_user_meta']);
});
