<?php

namespace DaHannesConnector\includes;

use DaHannesConnector\DahannesConnector;

class AdminSettingsPage
{
    public static function handle_actions(): void
    {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            return;
        }

        // Handle the main settings page submission
        if (isset($_POST['dahannes_server_url'], $_POST['api_key'])) {
            $server_url = sanitize_text_field($_POST['dahannes_server_url']);
            $api_key = sanitize_text_field($_POST['api_key']);

            update_option('dahannes_server_url', $server_url);
            update_option('api_key', $api_key);
        }

        DahannesConnectorUtils::update_info_from_dahannes_api();
    }

    public static function render_page(): void
    {
        static::handle_actions();

        // Get the connection status and information
        $api_call_success = get_option('dahannes_api_success', false);
        $info_data = get_option('dahannes_info', '');

        ?>
        <div class="wrap">
            <h1><?php _e('Dahannes Connector Einstellungen', 'dahannes-connector'); ?></h1>

            <!-- Display Connection Status -->
            <h2><?php _e('Connection Status', 'dahannes-connector'); ?></h2>
            <?php
            if ($api_call_success) {
                echo '<div style="color: green;">' . __('Connection was successful!', 'dahannes-connector') . '</div>';
            } else {
                $error_message = get_option('dahannes_api_error', __('Unknown error occurred', 'dahannes-connector'));
                echo '<div style="color: red;">' . __('Connection failed:', 'dahannes-connector') . ' ' . esc_html($error_message) . '</div>';

                // Define app type friendly names
                $app_type_labels = [
                    'association' => __('Vereinsshop', 'dahannes-connector'),
                    'commercial' => __('GmbH Shop', 'dahannes-connector'),
                    'partner' => __('Partnershop', 'dahannes-connector')
                ];

                // If successful, decode the "info" data and display app info
                if (isset($info_data['name'])) {
                    echo '<p><strong>' . __('App:', 'dahannes-connector') . '</strong> ' .
                        esc_html($info_data['name']);
                    if (isset($info_data['url'])) {
                        echo ' (' . esc_html($info_data['url']) . ')';
                    }
                    echo '</p>';
                }

                // Display shop type with friendly name
                if (isset($info_data['app_type'])) {
                    $app_type = $info_data['app_type'];
                    $friendly_name = $app_type_labels[$app_type] ?? $app_type;
                    echo '<p><strong>' . __('Shop Typ:', 'dahannes-connector') . '</strong> ' . esc_html($friendly_name) . '</p>';
                }

                // Display bonus percentage for all shop types
                if (isset($info_data['bonus_percentage'])) {
                    echo '<p><strong>' . __('Bonus Prozentsatz:', 'dahannes-connector') . '</strong> ' . esc_html($info_data['bonus_percentage']) . '%</p>';
                }

            }

            // Display active status regardless of connection success
            if (isset($info_data['is_active'])) {
                $active_status = $info_data['is_active'] ?
                    '<span style="color: green;">' . __('Active', 'dahannes-connector') . '</span>' :
                    '<span style="color: red;">' . __('Inactive', 'dahannes-connector') . '</span>';

                echo '<p><strong>' . __('Status:', 'dahannes-connector') . '</strong> ' . $active_status . '</p>';

                if (!$info_data['is_active']) {
                    echo '<div class="notice notice-warning inline"><p>' .
                        __('Warning: Your account is inactive. Bonuses will not be submitted.', 'dahannes-connector') .
                        '</p></div>';
                }
            }

            // Debug information moved to the Debug page
            ?>

            <form method="post" action="<?php echo esc_url(admin_url('admin.php?page=dahannes-connector')); ?>">
                <table class="form-table">
                    <tr valign="top">
                        <th scope="row"><?php _e('Server URL', 'dahannes-connector'); ?></th>
                        <td>
                            <input type="text" name="dahannes_server_url"
                                   value="<?php echo esc_attr(get_option('dahannes_server_url')); ?>"/>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row"><?php _e('API Key', 'dahannes-connector'); ?></th>
                        <td>
                            <input type="text" name="api_key" value="<?php echo esc_attr(get_option('api_key')); ?>"/>
                        </td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }

    public static function init(): void
    {
        add_action('admin_menu', function () {
            add_menu_page(
                __('Connection Settings', 'dahannes-connector'),
                __('DaHannes', 'dahannes-connector'),
                'manage_dahannes_connector',
                'dahannes-connector',
                [AdminSettingsPage::class, 'render_page'],
                DahannesConnector::get_menu_icon(),
            );
            add_submenu_page(
                'dahannes-connector',
                __('Connection Settings', 'dahannes-connector'),
                __('Verbindung', 'dahannes-connector'),
                'manage_dahannes_connector',
                'dahannes-connector',
                [AdminSettingsPage::class, 'render_page'],
            );
        });
    }
}
