<?php

namespace DaHannesConnector\includes;

class ApiClient
{
    private string $server_url;
    private string $api_key;
    private string $plugin_version;

    private function get_default_args(): array 
    {
        $is_development = (defined('WP_DEBUG') && WP_DEBUG) || (strpos($this->server_url, 'localhost') !== false);
        
        // Add debug logging for development
        if ($is_development) {
            DahannesLogger::log('debug', 'Using development settings', [
                'plugin_version' => $this->plugin_version,
                'server_url' => $this->server_url
            ]);
        }

        $headers = [
            'Authorization' => 'Bearer ' . trim($this->api_key), // Add trim() to remove any whitespace
            'Accept' => 'application/json',
            'X-DaHannes-Plugin-Version' => \DaHannesConnector\PLUGIN_VERSION // Use the constant directly
        ];
        
        return [
            'headers' => $headers,
            'timeout' => 5,
            'connect_timeout' => 5,
            'httpversion' => '1.1',
            'sslverify' => !$is_development
        ];
    }

    public function __construct(string $plugin_version = '1.0.0')
    {
        // Check if server_url and API key are set
        $this->server_url = get_option('dahannes_server_url');
        $this->api_key = get_option('api_key');
        $this->plugin_version = $plugin_version;
    }

    public function api_credentials_available(): bool
    {
        if (!$this->server_url || !$this->api_key) {
            $context = DahannesLogger::enrich_api_context([
                'server_url_set' => !empty($this->server_url),
                'api_key_set' => !empty($this->api_key),
                'server_url_length' => strlen($this->server_url ?? ''),
                'api_key_length' => strlen($this->api_key ?? ''),
            ], $this->server_url, $this->api_key);
            
            DahannesLogger::log('error', 'API credentials not available', $context);
            return false;
        }

        // Validate server URL format
        if (!filter_var($this->server_url, FILTER_VALIDATE_URL)) {
            $context = DahannesLogger::enrich_api_context([
                'server_url' => $this->server_url,
                'url_validation_error' => true,
                'url_components' => parse_url($this->server_url) ?: []
            ], $this->server_url, $this->api_key);
            
            DahannesLogger::log('error', 'Invalid server URL format', $context);
            return false;
        }

        return true;
    }

    public static function last_request_age_reached($seconds = 300): bool
    {
        // Get the last API call date
        $last_api_call_date = get_option('dahannes_last_api_call_date', 0);

        // Check if the last API call is within x seconds
        return (time() - $last_api_call_date) > $seconds;
    }

    public static function update_last_request_date(): void
    {
        update_option('dahannes_last_api_call_date', time());
    }

    public function get($endpoint, array $additional_arguments = []): ?array
    {
        if (!$this->api_credentials_available()) {
            return ['error' => 'API credentials not available or invalid'];
        }

        $a = array_merge_recursive($this->get_default_args(), $additional_arguments);
        $full_url = $this->server_url . $endpoint;
        $start_time = microtime(true);

        $result = wp_remote_get($full_url, $a);
        $response_time_ms = round((microtime(true) - $start_time) * 1000, 2);

        if (is_wp_error($result)) {
            $error_message = $result->get_error_message();
            $error_code = $result->get_error_code();
            
            $context = DahannesLogger::enrich_api_context([
                'endpoint' => $endpoint,
                'full_url' => $full_url,
                'error_code' => $error_code,
                'error_message' => $error_message,
                'response_time_ms' => $response_time_ms,
                'request_args' => array_intersect_key($a, array_flip(['timeout', 'connect_timeout', 'httpversion', 'sslverify'])),
                'method' => 'GET'
            ], $this->server_url, $this->api_key);
            
            DahannesLogger::log('error', 'GET request failed with WP_Error', $context);
            
            // Check if it's a host resolution error
            if (strpos($error_message, 'Could not resolve host') !== false) {
                return ['error' => __('Die Server-URL ist ungültig oder nicht erreichbar. Bitte überprüfen Sie die Einstellung.', 'dahannes-connector')];
            }
            return ['error' => $error_message];
        }

        $response_code = wp_remote_retrieve_response_code($result);
        $body = wp_remote_retrieve_body($result);
        $response_headers = wp_remote_retrieve_headers($result);

        if ($response_code >= 400) {
            $error_message = json_decode($body, true)['message'] ?? 'Unknown error';
            
            $context = DahannesLogger::enrich_api_context([
                'endpoint' => $endpoint,
                'full_url' => $full_url,
                'http_status' => $response_code,
                'error_message' => $error_message,
                'response_time_ms' => $response_time_ms,
                'response_body' => substr($body, 0, 500), // First 500 chars for debugging
                'response_headers' => array_intersect_key($response_headers->getAll(), array_flip(['content-type', 'x-ratelimit-remaining', 'retry-after'])),
                'method' => 'GET'
            ], $this->server_url, $this->api_key);
            
            DahannesLogger::log('error', 'GET request failed with HTTP error', $context);
            
            if ($response_code === 401) {
                return ['error' => __('Die API-Authentifizierung ist fehlgeschlagen. Bitte überprüfen Sie den API-Schlüssel.', 'dahannes-connector')];
            }
            if ($response_code === 426) {
                // Parse version information from error message
                if (preg_match('/Client version ([\d.]+) is not supported\. Minimum required version is ([\d.]+)/', $error_message, $matches)) {
                    return ['error' => sprintf(
                        __('Your DaHannes Connector plugin is outdated. Please update to version %s or higher to continue using the service.', 'dahannes-connector'),
                        $matches[2]
                    )];
                }
            }
            return ['error' => sprintf("HTTP %d: %s", $response_code, $error_message)];
        }

        if (!$body) {
            $context = DahannesLogger::enrich_api_context([
                'endpoint' => $endpoint,
                'full_url' => $full_url,
                'http_status' => $response_code,
                'response_time_ms' => $response_time_ms,
                'method' => 'GET'
            ], $this->server_url, $this->api_key);
            
            DahannesLogger::log('error', 'GET request returned empty response body', $context);
            return ['error' => 'Empty response from server'];
        }

        // Log successful requests in debug mode
        if (DahannesLogger::get_current_environment() === 'development') {
            $context = DahannesLogger::enrich_api_context([
                'endpoint' => $endpoint,
                'http_status' => $response_code,
                'response_time_ms' => $response_time_ms,
                'response_size_bytes' => strlen($body),
                'method' => 'GET'
            ], $this->server_url, $this->api_key);
            
            DahannesLogger::log('debug', 'GET request completed successfully', $context);
        }

        return ['success' => json_decode($body, true)];
    }
    
    public function get_bonuses(array $params = []): ?array
    {
        $endpoint = '/api/client/shop-bonuses';
        $additional_args = [];
        
        // Add query parameters if provided
        if (!empty($params)) {
            $endpoint .= '?' . http_build_query($params);
        }
        
        return $this->get($endpoint, $additional_args);
    }

    public function post($endpoint, array $body, array $additional_arguments = []): int
    {
        if (!$this->api_credentials_available()) {
            return 0;
        }

        $args = $this->get_default_args();
        if (!empty($additional_arguments)) {
            $args = array_merge($args, $additional_arguments);
        }
        $args['body'] = $body;  // Ensure body is set last to prevent conflicts

        $full_url = $this->server_url . $endpoint;
        $start_time = microtime(true);
        
        $result = wp_remote_post($full_url, $args);
        $response_time_ms = round((microtime(true) - $start_time) * 1000, 2);

        if (is_wp_error($result)) {
            $error_message = $result->get_error_message();
            $error_code = $result->get_error_code();
            
            $context = DahannesLogger::enrich_api_context([
                'endpoint' => $endpoint,
                'full_url' => $full_url,
                'error_code' => $error_code,
                'error_message' => $error_message,
                'response_time_ms' => $response_time_ms,
                'request_body_size' => strlen(json_encode($body)),
                'request_args' => array_intersect_key($args, array_flip(['timeout', 'connect_timeout', 'httpversion', 'sslverify'])),
                'method' => 'POST'
            ], $this->server_url, $this->api_key);
            
            DahannesLogger::log('error', 'POST request failed with WP_Error', $context);
            return 0;
        }

        $response_code = (int)wp_remote_retrieve_response_code($result);
        $response_body = wp_remote_retrieve_body($result);
        $response_headers = wp_remote_retrieve_headers($result);
        
        if ($response_code >= 400) {
            $context = DahannesLogger::enrich_api_context([
                'endpoint' => $endpoint,
                'full_url' => $full_url,
                'http_status' => $response_code,
                'response_time_ms' => $response_time_ms,
                'request_body_size' => strlen(json_encode($body)),
                'response_body' => substr($response_body, 0, 500), // First 500 chars
                'response_headers' => array_intersect_key($response_headers->getAll(), array_flip(['content-type', 'x-ratelimit-remaining', 'retry-after'])),
                'method' => 'POST'
            ], $this->server_url, $this->api_key);
            
            DahannesLogger::log('error', 'POST request failed with HTTP error', $context);
        } else {
            // Log successful POST requests in debug mode
            if (DahannesLogger::get_current_environment() === 'development') {
                $context = DahannesLogger::enrich_api_context([
                    'endpoint' => $endpoint,
                    'http_status' => $response_code,
                    'response_time_ms' => $response_time_ms,
                    'request_body_size' => strlen(json_encode($body)),
                    'response_size_bytes' => strlen($response_body),
                    'method' => 'POST'
                ], $this->server_url, $this->api_key);
                
                DahannesLogger::log('debug', 'POST request completed successfully', $context);
            }
        }

        return $response_code;
    }


    public function send_order_list(array $orders): bool
    {
        // Validate batch size
        if (count($orders) > 50) {
            $context = DahannesLogger::enrich_api_context([
                'order_count' => count($orders),
                'max_allowed' => 50,
                'order_ids' => array_keys($orders),
                'validation_error' => 'batch_size_exceeded'
            ], $this->server_url, $this->api_key);

            DahannesLogger::log('error', 'Batch size exceeds maximum of 50 orders', $context);
            return false;
        }

        // Prepare order data for submission
        $order_ids = array_keys($orders);
        $order_data = ['orders' => $orders];

        $code = $this->post('/api/orders', $order_data, ['timeout' => 60]);

        if ($code !== 200) {
            $context = DahannesLogger::enrich_api_context([
                'endpoint' => '/api/orders',
                'expected_status' => 200,
                'actual_status' => $code,
                'order_count' => count($orders),
                'order_ids' => $order_ids,
                'request_timeout' => 60
            ], $this->server_url, $this->api_key);

            DahannesLogger::log('error', 'Failed to send order list - unexpected response code', $context);
            return false;
        }

        // Log successful batch submission
        $context = DahannesLogger::enrich_api_context([
            'endpoint' => '/api/orders',
            'order_count' => count($orders),
            'order_ids' => $order_ids,
            'http_status' => $code
        ], $this->server_url, $this->api_key);

        DahannesLogger::log('info', 'Successfully sent order list to API', $context);

        return true;
    }
}
