<?php

namespace DaHannesConnector\includes;

class CronScheduler {
    private static $last_run_option = 'dh_last_sync_time';
    private static $interval = 60; // 60 seconds = 1 minute

    public static function init(): void 
    {
        // Add our check on every page load
        add_action('init', [self::class, 'check_and_run']);

        // Add manual trigger capability with nonce verification
        add_action('admin_post_trigger_dahannes_sync', [self::class, 'handle_manual_trigger']);
        add_action('admin_notices', [self::class, 'show_sync_notices']);
    }

    public static function handle_manual_trigger(): void 
    {
        if (!current_user_can('manage_dahannes_connector')) {
            wp_die(__('You do not have permission to do this.', 'dahannes-connector'));
        }

        check_admin_referer('trigger_dahannes_sync');

        // Rate limiting check
        if (get_transient('dahannes_manual_sync_lock')) {
            wp_redirect(add_query_arg('dahannes_sync_error', 'rate_limit', wp_get_referer()));
            exit;
        }

        set_transient('dahannes_manual_sync_lock', true, 60);

        try {
            self::run_sync();
            wp_redirect(add_query_arg('dahannes_sync_success', '1', wp_get_referer()));
        } catch (\Exception $e) {
            wp_redirect(add_query_arg('dahannes_sync_error', 'failed', wp_get_referer()));
        }
        exit;
    }

    public static function show_sync_notices(): void 
    {
        if (!empty($_GET['dahannes_sync_error'])) {
            $message = $_GET['dahannes_sync_error'] === 'rate_limit' 
                ? __('Please wait a minute before triggering another sync.', 'dahannes-connector')
                : __('Sync failed. Please check the logs for details.', 'dahannes-connector');
            
            echo '<div class="notice notice-error"><p>' . esc_html($message) . '</p></div>';
        }

        if (!empty($_GET['dahannes_sync_success'])) {
            echo '<div class="notice notice-success"><p>' . 
                 __('Manual sync completed successfully.', 'dahannes-connector') . 
                 '</p></div>';
        }
    }

    public static function check_and_run(): void 
    {
        $last_run = get_option(self::$last_run_option, 0);
        $now = time();
        
        if (($now - $last_run) >= self::$interval) {
            self::run_sync();
        }
    }

    private static function run_sync(): void 
    {
        try {
            DahannesConnectorUtils::update_info_from_dahannes_api();
            DahannesOrderManager::send_orders_to_server();
            
            // Update last run time only on successful completion
            update_option(self::$last_run_option, time());
            
        } catch (\Exception $e) {
            DahannesLogger::log('error', 'Sync error: ' . $e->getMessage());
        }
    }
}
