<?php

namespace DaHannesConnector\includes;

class DahannesAdminUI
{
    public function __construct()
    {
        add_action('admin_menu', [$this, 'add_status_dashboard']);
        add_filter('bulk_actions-edit-shop_order', [$this, 'add_bulk_actions']);
        add_filter('woocommerce_order_actions', [$this, 'add_order_actions']);
        add_action('admin_post_resync_with_dahannes', [$this, 'handle_resync']);

        // Add these new hooks
        add_action('handle_bulk_actions-edit-shop_order', [$this, 'handle_bulk_resync'], 10, 3);
        add_action('admin_notices', [$this, 'show_bulk_resync_notices']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        add_action('woocommerce_order_action_resync_single_order', [$this, 'handle_single_order_resync']);
        add_action('admin_post_test_sentry_logging', [$this, 'handle_test_sentry_logging']);
    }

    public function add_status_dashboard(): void
    {
        // Get shop type
        $info_data = get_option('dahannes_info', []);
        $shop_type = $info_data['app_type'] ?? '';

        // Only show debug page for association or commercial shop types
        if ($shop_type === 'association' || $shop_type === 'commercial') {
            add_submenu_page(
                'dahannes-connector',
                __('DaHannes Debug', 'dahannes-connector'),
                __('Debug', 'dahannes-connector'),
                'manage_dahannes_connector',
                'dahannes-status',
                [$this, 'render_status_page']
            );
        }

        // Add Bonuses page only if connection is successful
        $api_success = get_option('dahannes_api_success', false);
        if ($api_success) {
            add_submenu_page(
                'dahannes-connector',
                __('Bonuses', 'dahannes-connector'),
                __('Bonuses', 'dahannes-connector'),
                'manage_dahannes_connector',
                'dahannes-bonuses',
                [$this, 'render_bonuses_page']
            );
        }
    }

    public function add_bulk_actions($actions): array
    {
        $actions['resync_with_dahannes'] = __('Resync with DaHannes', 'dahannes-connector');
        return $actions;
    }

    public function add_order_actions($actions): array
    {
        $actions['resync_single_order'] = __('Resync with DaHannes', 'dahannes-connector');
        return $actions;
    }

    public function handle_bulk_resync($redirect_to, $action, $post_ids)
    {
        if ($action !== 'resync_with_dahannes') {
            return $redirect_to;
        }

        // Rate limiting check
        $last_resync = get_transient('dahannes_last_bulk_resync');
        if ($last_resync) {
            $redirect_to = add_query_arg('dahannes_resync_error', 'rate_limit', $redirect_to);
            return $redirect_to;
        }

        // Fast atomic reset using direct database query
        $reset_count = $this->atomic_reset_sync_status($post_ids);

        // Set rate limiting transient
        set_transient('dahannes_last_bulk_resync', time(), 60); // 1 minute cooldown

        $redirect_to = add_query_arg([
            'dahannes_resync_done' => count($post_ids),
            'dahannes_resync_success' => $reset_count,
            'dahannes_resync_failed' => 0
        ], $redirect_to);

        return $redirect_to;
    }

    public function show_bulk_resync_notices()
    {
        if (!empty($_GET['dahannes_resync_error']) && $_GET['dahannes_resync_error'] === 'rate_limit') {
            echo '<div class="notice notice-error"><p>' .
                __('Please wait a minute before performing another bulk resync.', 'dahannes-connector') .
                '</p></div>';
            return;
        }

        if (!empty($_GET['dahannes_resync_done'])) {
            $total = intval($_GET['dahannes_resync_done']);
            $success = intval($_GET['dahannes_resync_success']);
            $failed = intval($_GET['dahannes_resync_failed']);

            $message = sprintf(
                __('Resync completed. Successfully processed %d orders, %d failed.', 'dahannes-connector'),
                $success,
                $failed
            );

            $class = ($failed > 0) ? 'notice-warning' : 'notice-success';
            echo "<div class='notice {$class}'><p>{$message}</p></div>";
        }

        if (!empty($_GET['sentry_test']) && $_GET['sentry_test'] === 'success') {
            echo '<div class="notice notice-success"><p>' .
                __('Sentry test logs have been sent! Check your Sentry dashboard for the test messages (info, warning, and error levels).', 'dahannes-connector') .
                '</p></div>';
        }
    }

    public function enqueue_admin_scripts()
    {
        wp_enqueue_script(
            'dahannes-admin',
            plugins_url('js/admin.js', dirname(__FILE__)),
            ['jquery'],
            '1.0.0',
            true
        );

        wp_localize_script('dahannes-admin', 'dahannesAdmin', [
            'confirmResync' => __('Are you sure you want to resync these orders with DaHannes?', 'dahannes-connector'),
            'processing' => __('Processing...', 'dahannes-connector')
        ]);

        // Add datepicker scripts for bonuses page
        if (isset($_GET['page']) && $_GET['page'] === 'dahannes-bonuses') {
            wp_enqueue_script('jquery-ui-datepicker');
            wp_enqueue_style('jquery-ui', 'https://code.jquery.com/ui/1.13.2/themes/smoothness/jquery-ui.css');
        }
    }

    public function render_status_page(): void
    {
        ?>
        <div class="wrap">
            <h1><?php _e('DaHannes Debug', 'dahannes-connector'); ?></h1>

            <div class="manual-sync-section" style="margin: 20px 0;">
                <form method="post" action="<?php echo admin_url('admin-post.php'); ?>">
                    <?php wp_nonce_field('trigger_dahannes_sync'); ?>
                    <input type="hidden" name="action" value="trigger_dahannes_sync">
                    <button type="submit" class="button button-primary">
                        <?php _e('Trigger Manual Sync', 'dahannes-connector'); ?>
                    </button>
                </form>
            </div>

            <!-- Debug Information Section -->
            <div class="debug-info"
                 style="margin-top: 20px; padding: 10px; background: #f5f5f5; border: 1px solid #ccc;">
                <h3><?php _e('Debug Information', 'dahannes-connector'); ?></h3>

                <?php
                // Show last API call details
                $last_api_call = get_option('dahannes_last_api_call_date', 0);
                echo '<p><strong>' . __('Last API Call:', 'dahannes-connector') . '</strong> ' .
                    ($last_api_call ? wp_date('Y-m-d H:i:s', $last_api_call) : __('Never', 'dahannes-connector')) . '</p>';

                // Show current API settings
                echo '<p><strong>' . __('Current Server URL:', 'dahannes-connector') . '</strong> ' .
                    esc_html(get_option('dahannes_server_url', __('Not set', 'dahannes-connector'))) . '</p>';

                // Show if API key is set (but not the actual key)
                $api_key = get_option('api_key', '');
                echo '<p><strong>' . __('API Key Status:', 'dahannes-connector') . '</strong> ' .
                    ($api_key ? __('Set', 'dahannes-connector') : __('Not set', 'dahannes-connector')) . '</p>';

                // Show log file status
                $log_file = WP_CONTENT_DIR . '/dahannes-connector.log';
                echo '<p><strong>' . __('Log File:', 'dahannes-connector') . '</strong> ' . $log_file . '<br>';
                echo __('Exists:', 'dahannes-connector') . ' ' . (file_exists($log_file) ? __('Yes', 'dahannes-connector') : __('No', 'dahannes-connector')) . '<br>';
                echo __('Writable:', 'dahannes-connector') . ' ' . (is_writable($log_file) ? __('Yes', 'dahannes-connector') : __('No', 'dahannes-connector')) . '</p>';

                // Show WordPress debug settings
                echo '<p><strong>' . __('WordPress Debug Settings:', 'dahannes-connector') . '</strong><br>';
                echo 'WP_DEBUG: ' . (defined('WP_DEBUG') && WP_DEBUG ? __('Enabled', 'dahannes-connector') : __('Disabled', 'dahannes-connector')) . '<br>';
                echo 'WP_DEBUG_LOG: ' . (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG ? __('Enabled', 'dahannes-connector') : __('Disabled', 'dahannes-connector')) . '</p>';

                // Show Sentry status
                echo '<p><strong>' . __('Sentry Configuration:', 'dahannes-connector') . '</strong><br>';
                $sentry_disabled = defined('DAHANNES_DISABLE_SENTRY') && DAHANNES_DISABLE_SENTRY;
                echo 'DAHANNES_DISABLE_SENTRY: ' . ($sentry_disabled ? __('Enabled (Sentry disabled)', 'dahannes-connector') : __('Not set (Sentry enabled by default)', 'dahannes-connector')) . '<br>';
                
                if (!$sentry_disabled) {
                    $custom_dsn = defined('DAHANNES_SENTRY_DSN') && !empty(DAHANNES_SENTRY_DSN);
                    echo 'Sentry DSN: ' . ($custom_dsn ? __('Custom DSN configured', 'dahannes-connector') : __('Using default DSN', 'dahannes-connector')) . '<br>';
                    
                    $environment = DahannesLogger::get_current_environment();
                    echo 'Environment: ' . esc_html($environment) . '<br>';
                    
                    $custom_env = defined('DAHANNES_ENVIRONMENT');
                    echo 'Environment Detection: ' . ($custom_env ? __('Manual override', 'dahannes-connector') : __('Auto-detected', 'dahannes-connector')) . '<br>';
                    
                    $sample_rate = defined('DAHANNES_SENTRY_SAMPLE_RATE') ? DAHANNES_SENTRY_SAMPLE_RATE : ($environment === 'production' ? 0.1 : 1.0);
                    echo 'Sample Rate: ' . ($sample_rate * 100) . '%<br>';
                    
                    $context = DahannesLogger::get_current_installation_context();
                    echo 'Installation ID: ' . esc_html(parse_url($context['site_url'], PHP_URL_HOST)) . '<br>';
                    
                    // Add test button for Sentry
                    echo '<br><form method="post" action="' . admin_url('admin-post.php') . '" style="display: inline;">';
                    echo '<input type="hidden" name="action" value="test_sentry_logging">';
                    wp_nonce_field('test_sentry_logging', 'test_sentry_nonce');
                    echo '<button type="submit" class="button button-secondary">' . __('Test Sentry Logging', 'dahannes-connector') . '</button>';
                    echo '</form>';
                } else {
                    echo __('Sentry logging is disabled. Remove DAHANNES_DISABLE_SENTRY=true from wp-config.php to enable', 'dahannes-connector') . '<br>';
                }
                echo '</p>';
                ?>
            </div>

            <h2><?php _e('Recent Activity', 'dahannes-connector'); ?></h2>
            <table class="widefat">
                <thead>
                <tr>
                    <th><?php _e('Time', 'dahannes-connector'); ?></th>
                    <th><?php _e('Level', 'dahannes-connector'); ?></th>
                    <th><?php _e('Message', 'dahannes-connector'); ?></th>
                </tr>
                </thead>
                <tbody>
                <?php foreach (DahannesLogger::get_recent_logs() as $log): ?>
                    <tr>
                        <td><?php echo esc_html($log['timestamp']); ?></td>
                        <td><?php echo esc_html(strtoupper($log['level'])); ?></td>
                        <td><?php echo esc_html($log['message']); ?></td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    public function handle_single_order_resync($order)
    {
        if (!$order instanceof \WC_Order) {
            return;
        }

        // Rate limiting check
        if (get_transient('dahannes_single_resync_lock')) {
            return;
        }

        set_transient('dahannes_single_resync_lock', true, 60);

        try {
            // Fast atomic reset for single order
            $reset_count = $this->atomic_reset_sync_status([$order->get_id()]);

            if ($reset_count > 0) {
                $order->add_order_note(__('Order manually resynced with DaHannes.', 'dahannes-connector'));
            } else {
                $order->add_order_note(__('Failed to resync order with DaHannes.', 'dahannes-connector'));
            }
        } catch (\Exception $e) {
            $order->add_order_note(__('Error resyncing order with DaHannes: ', 'dahannes-connector') . $e->getMessage());
        }
    }

    public function render_bonuses_page(): void
    {
        $client = new ApiClient();
        $message = '';
        $orders = [];
        $pagination = [];

        // Get sanitized filter values
        $filters = $this->get_filter_values();

        // Prepare API parameters
        $params = $this->prepare_api_params($filters);

        // Fetch orders from API
        if ($client->api_credentials_available()) {
            $response = $client->get_bonuses($params);

            if (isset($response['error'])) {
                $message = '<div class="notice notice-error"><p>' . esc_html($response['error']) . '</p></div>';
            } elseif (isset($response['success'])) {
                $data = $response['success'];
                $orders = $data['data'] ?? [];
                $pagination = [
                    'current_page' => $data['meta']['current_page'] ?? 1,
                    'last_page' => $data['meta']['total_pages'] ?? 1,
                    'total' => $data['meta']['total_records'] ?? 0
                ];
            }
        } else {
            $message = '<div class="notice notice-error"><p>' .
                __('API credentials are not available. Please check your connection settings.', 'dahannes-connector') .
                '</p></div>';
        }

        ?>
        <div class="wrap" style="max-width: 1000px;">
            <h1><?php _e('Overview of Bonuses', 'dahannes-connector'); ?></h1>

            <?php echo $message; ?>

            <?php $this->render_filter_section($filters); ?>
            <div style="clear: both; margin-bottom: 20px;"></div>

            <!-- Summary Information -->
            <?php if (isset($response['success']) && isset($response['success']['meta'])): ?>
                <?php $meta = $response['success']['meta']; ?>
                <div class="dahannes-summary"
                     style="margin: 20px 0; padding: 15px; background-color: #f8f8f8; border-left: 4px solid #2271b1; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
                    <div class="dahannes-summary-grid"
                         style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); grid-gap: 15px;">
                        
                        <!-- Total Bonuses -->
                        <div class="summary-item" style="text-align: center;">
                            <p class="big-number" style="font-size: 20px; font-weight: bold; margin: 0 0 5px 0;">
                                <?php echo isset($meta['items_count']) ? number_format($meta['items_count']) : '0'; ?>
                            </p>
                            <p style="margin: 0; font-size: 14px;"><?php _e('Total Bonuses', 'dahannes-connector'); ?></p>
                        </div>

                        <!-- Cleared Bonuses -->
                        <?php if (isset($meta['status_counts']['cleared'])): ?>
                        <div class="summary-item" style="text-align: center;">
                            <p class="big-number" style="font-size: 20px; font-weight: bold; margin: 0 0 5px 0; color: green;">
                                <?php echo number_format($meta['status_counts']['cleared']); ?>
                            </p>
                            <p style="margin: 0; font-size: 14px;"><?php _e('Cleared', 'dahannes-connector'); ?></p>
                        </div>
                        <?php endif; ?>

                        <!-- Pending Bonuses -->
                        <?php if (isset($meta['status_counts']['pending'])): ?>
                        <div class="summary-item" style="text-align: center;">
                            <p class="big-number" style="font-size: 20px; font-weight: bold; margin: 0 0 5px 0; color: orange;">
                                <?php echo number_format($meta['status_counts']['pending']); ?>
                            </p>
                            <p style="margin: 0; font-size: 14px;"><?php _e('Pending', 'dahannes-connector'); ?></p>
                        </div>
                        <?php endif; ?>

                        <!-- Cancelled Bonuses -->
                        <?php if (isset($meta['status_counts']['cancelled'])): ?>
                        <div class="summary-item" style="text-align: center;">
                            <p class="big-number" style="font-size: 20px; font-weight: bold; margin: 0 0 5px 0; color: red;">
                                <?php echo number_format($meta['status_counts']['cancelled']); ?>
                            </p>
                            <p style="margin: 0; font-size: 14px;"><?php _e('Cancelled', 'dahannes-connector'); ?></p>
                        </div>
                        <?php endif; ?>

                        <!-- Total Cleared Bonus Amount -->
                        <div class="summary-item" style="text-align: center;">
                            <p class="big-number" style="font-size: 20px; font-weight: bold; margin: 0 0 5px 0; color: green;">
                                <?php
                                // API v2.4.0: total_bonus_amount includes only cleared bonuses
                                if (isset($meta['total_bonus_amount'])) {
                                    echo number_format((float)$meta['total_bonus_amount'], 2, ',', '.') . ' €';
                                } else {
                                    echo '0,00 €';
                                }
                                ?>
                            </p>
                            <p style="margin: 0; font-size: 14px;"><?php _e('Cleared Bonus Amount', 'dahannes-connector'); ?></p>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Orders Table -->
            <table class="widefat striped">
                <thead>
                <tr>
                    <th><?php _e('Date', 'dahannes-connector'); ?></th>
                    <th><?php _e('Bonus ID', 'dahannes-connector'); ?></th>
                    <th><?php _e('Order', 'dahannes-connector'); ?></th>
                    <th><?php _e('Bonus Amount', 'dahannes-connector'); ?></th>
                    <th><?php _e('Member Code', 'dahannes-connector'); ?></th>
                    <th><?php _e('Status', 'dahannes-connector'); ?></th>
                </tr>
                </thead>
                <tbody>
                <?php if (empty($orders)): ?>
                    <tr>
                        <td colspan="6"><?php _e('No bonuses found.', 'dahannes-connector'); ?></td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($orders as $bonus): ?>
                        <tr>
                            <td>
                                <?php
                                if (!empty($bonus['created_at'])) {
                                    // Convert ISO date to timestamp and format it according to WP settings
                                    $timestamp = strtotime($bonus['created_at']);
                                    echo date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $timestamp);
                                }
                                ?>
                            </td>
                            <td>
                                <?php echo esc_html($bonus['id'] ?? ''); ?>
                            </td>
                            <td>
                                <?php
                                // API v2.3.0: bonuses now have wc_order_id containing the WooCommerce order ID
                                $wc_order_id = $bonus['wc_order_id'] ?? null;
                                if ($wc_order_id && ($local_order = wc_get_order($wc_order_id))):
                                    // Get customer name and order number - format: Name (Order #123)
                                    $order_number = $local_order->get_order_number();

                                    // Get billing name if available, otherwise use "Guest"
                                    $billing_first_name = $local_order->get_billing_first_name();
                                    $billing_last_name = $local_order->get_billing_last_name();

                                    if (!empty($billing_first_name) || !empty($billing_last_name)) {
                                        $customer_name = trim($billing_first_name . ' ' . $billing_last_name);
                                    } else {
                                        $customer_name = __('Guest', 'dahannes-connector');
                                    }

                                    // Format exactly like WooCommerce: "#123 Max Mustermann"
                                    $order_title = sprintf(
                                        '#%s %s',
                                        $order_number,
                                        esc_html($customer_name)
                                    );
                                    ?>
                                    <a href="<?php echo admin_url('post.php?post=' . $wc_order_id . '&action=edit'); ?>">
                                        <?php echo $order_title; ?>
                                    </a>
                                <?php elseif ($wc_order_id): ?>
                                    <?php printf(__('Order #%s (not found locally)', 'dahannes-connector'), esc_html($wc_order_id)); ?>
                                <?php else: ?>
                                    <?php _e('Manual Bonus', 'dahannes-connector'); ?>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php
                                if (isset($bonus['amount'])) {
                                    // API v2.2.0 sends amounts in euros (float)
                                    echo number_format((float)$bonus['amount'], 2, ',', '.') . ' €';
                                } else {
                                    echo '-';
                                }
                                ?>
                            </td>
                            <td><?php echo esc_html($bonus['member_code'] ?? ''); ?></td>
                            <td>
                                <?php if (isset($bonus['status'])): ?>
                                    <?php
                                    $status_colors = [
                                        'cleared' => 'green',
                                        'pending' => 'orange',
                                        'cancelled' => 'red'
                                    ];
                                    $status_labels = [
                                        'cleared' => __('Bestätigt', 'dahannes-connector'),
                                        'pending' => __('Offen', 'dahannes-connector'),
                                        'cancelled' => __('Storniert', 'dahannes-connector')
                                    ];
                                    $color = $status_colors[$bonus['status']] ?? 'black';
                                    $label = $status_labels[$bonus['status']] ?? ucfirst($bonus['status']);
                                    ?>
                                    <span style="color:<?php echo $color; ?>;">
                                            <?php echo esc_html($label); ?>
                                        </span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
                </tbody>
            </table>

            <!-- Pagination -->
            <?php if (!empty($pagination) && $pagination['last_page'] > 1): ?>
                <div class="tablenav bottom">
                    <div class="tablenav-pages">
                        <span class="displaying-num">
                            <?php
                            printf(
                                _n('%s item', '%s items', $pagination['total'], 'dahannes-connector'),
                                number_format_i18n($pagination['total'])
                            );
                            ?>
                        </span>

                        <span class="pagination-links">
                            <?php
                            // First page
                            if ($pagination['current_page'] > 1) {
                                echo '<a class="first-page button" href="' .
                                    add_query_arg(['page_num' => 1]) .
                                    '"><span class="screen-reader-text">' .
                                    __('First page', 'dahannes-connector') .
                                    '</span><span aria-hidden="true">&laquo;</span></a>';

                                echo '<a class="prev-page button" href="' .
                                    add_query_arg(['page_num' => $pagination['current_page'] - 1]) .
                                    '"><span class="screen-reader-text">' .
                                    __('Previous page', 'dahannes-connector') .
                                    '</span><span aria-hidden="true">&lsaquo;</span></a>';
                            } else {
                                echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&laquo;</span>';
                                echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&lsaquo;</span>';
                            }

                            // Page numbers
                            echo '<span class="paging-input">' .
                                sprintf(
                                    _x('%1$s of %2$s', 'paging', 'dahannes-connector'),
                                    '<span class="current-page">' . $pagination['current_page'] . '</span>',
                                    '<span class="total-pages">' . $pagination['last_page'] . '</span>'
                                ) .
                                '</span>';

                            // Last page
                            if ($pagination['current_page'] < $pagination['last_page']) {
                                echo '<a class="next-page button" href="' .
                                    add_query_arg(['page_num' => $pagination['current_page'] + 1]) .
                                    '"><span class="screen-reader-text">' .
                                    __('Next page', 'dahannes-connector') .
                                    '</span><span aria-hidden="true">&rsaquo;</span></a>';

                                echo '<a class="last-page button" href="' .
                                    add_query_arg(['page_num' => $pagination['last_page']]) .
                                    '"><span class="screen-reader-text">' .
                                    __('Last page', 'dahannes-connector') .
                                    '</span><span aria-hidden="true">&raquo;</span></a>';
                            } else {
                                echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&rsaquo;</span>';
                                echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">&raquo;</span>';
                            }
                            ?>
                        </span>
                    </div>
                </div>
            <?php endif; ?>
        </div>

        <script>
            jQuery(document).ready(function ($) {
                $('.datepicker').datepicker({
                    dateFormat: 'dd.mm.yy',
                    changeMonth: true,
                    changeYear: true,
                    regional: 'de'
                });
            });
        </script>
        <?php
    }

    private function get_filter_values(): array
    {
        return [
            'start_date' => isset($_GET['start_date']) ? sanitize_text_field($_GET['start_date']) : '',
            'end_date' => isset($_GET['end_date']) ? sanitize_text_field($_GET['end_date']) : '',
            'member_code' => isset($_GET['member_code']) ? sanitize_text_field($_GET['member_code']) : '',
            'per_page' => isset($_GET['per_page']) ? (int)$_GET['per_page'] : 100,
            'page' => isset($_GET['page_num']) ? (int)$_GET['page_num'] : 1
        ];
    }

    private function prepare_api_params(array $filters): array
    {
        $params = [];

        if (!empty($filters['start_date'])) {
            $date = $this->validate_and_convert_date($filters['start_date']);
            if ($date) {
                $params['start_date'] = $date;
            }
        }

        if (!empty($filters['end_date'])) {
            $date = $this->validate_and_convert_date($filters['end_date']);
            if ($date) {
                $params['end_date'] = $date;
            }
        }

        if (!empty($filters['member_code'])) {
            $params['member_code'] = $filters['member_code'];
        }

        $params['per_page'] = min(500, max(5, $filters['per_page']));
        $params['page'] = $filters['page'];

        return $params;
    }

    private function validate_and_convert_date(string $date_string): ?string
    {
        $date = \DateTime::createFromFormat('d.m.Y', $date_string);
        return $date ? $date->format('Y-m-d') : null;
    }

    private function render_filter_section(array $filters): void
    {
        ?>
        <!-- Filter Form -->
        <div style="display: flex; flex-direction: column; gap: 0.7rem; margin-bottom: 1.5rem; margin-top: 1.5rem;">
            <div>
                <?php $this->render_quick_filters(); ?>
            </div>

            <form method="get" action="">
                <input type="hidden" name="page" value="dahannes-bonuses">

                <div class="alignleft actions" style="margin-bottom: 10px;">
                    <div style="display: flex; flex-wrap: wrap; align-items: center; gap: 10px;">
                        <div style="display: flex; align-items: center; gap: 5px;">
                            <label for="start_date"><?php _e('Start Date:', 'dahannes-connector'); ?></label>
                            <input type="text" id="start_date" name="start_date"
                                   value="<?php echo esc_attr($filters['start_date']); ?>"
                                   class="datepicker" placeholder="<?php echo date_i18n('d.m.Y'); ?>"
                                   style="width: 90px;">
                        </div>

                        <div style="display: flex; align-items: center; gap: 5px;">
                            <label for="end_date"><?php _e('End Date:', 'dahannes-connector'); ?></label>
                            <input type="text" id="end_date" name="end_date"
                                   value="<?php echo esc_attr($filters['end_date']); ?>"
                                   class="datepicker" placeholder="<?php echo date_i18n('d.m.Y'); ?>"
                                   style="width: 90px;">
                        </div>

                        <div style="display: flex; align-items: center; gap: 5px;">
                            <label for="member_code"><?php _e('Member Code:', 'dahannes-connector'); ?></label>
                            <input type="text" id="member_code" name="member_code"
                                   value="<?php echo esc_attr($filters['member_code']); ?>"
                                   placeholder="ABC123" style="width: 80px;">
                        </div>

                        <div style="display: flex; align-items: center; gap: 5px;">
                            <label for="per_page"><?php _e('Per Page:', 'dahannes-connector'); ?></label>
                            <select id="per_page" name="per_page" style="width: 60px;">
                                <?php foreach ([15, 25, 50, 100, 200, 500] as $option): ?>
                                    <option value="<?php echo $option; ?>" <?php selected($filters['per_page'], $option); ?>>
                                        <?php echo $option; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div style="display: flex; align-items: center; gap: 5px;">
                            <input type="submit" class="button"
                                   value="<?php _e('Filter', 'dahannes-connector'); ?>">

                            <?php if ($this->has_active_filters($filters)): ?>
                                <a href="<?php echo admin_url('admin.php?page=dahannes-bonuses'); ?>"
                                   class="button">
                                    <?php _e('Clear Filters', 'dahannes-connector'); ?>
                                </a>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </form>

            <div style="clear: both;"></div>
        </div>
        <div style="clear: both; margin-bottom: 20px;"></div>
        <?php
    }

    private function has_active_filters(array $filters): bool
    {
        return !empty($filters['start_date']) ||
            !empty($filters['end_date']) ||
            !empty($filters['member_code']) ||
            $filters['per_page'] !== 100;
    }

    private function render_quick_filters(): void
    {
        $quick_filters = $this->get_quick_filter_options();
        ?>
        <!-- Quick Filters -->
        <span style="font-weight: bold; margin-right: 10px;"><?php _e('Quick Filters', 'dahannes-connector'); ?>:</span>
        <?php foreach ($quick_filters as $filter): ?>
        <a href="<?php echo esc_url($filter['url']); ?>" class="button" style="margin-right: 5px;">
            <?php echo esc_html($filter['label']); ?>
        </a>
    <?php endforeach; ?>
        <?php
    }

    private function get_quick_filter_options(): array
    {
        $base_args = ['page' => 'dahannes-bonuses'];

        // Get month names using WordPress localization
        $current_month = date_i18n('F Y');
        $previous_month = date_i18n('F Y', strtotime('first day of last month'));
        $two_months_ago = date_i18n('F Y', strtotime('first day of -2 month'));

        return [
            [
                'label' => $current_month,
                'url' => add_query_arg(array_merge($base_args, [
                    'start_date' => date('01.m.Y'),
                    'end_date' => date('t.m.Y')
                ]))
            ],
            [
                'label' => $previous_month,
                'url' => add_query_arg(array_merge($base_args, [
                    'start_date' => date('01.m.Y', strtotime('first day of last month')),
                    'end_date' => date('t.m.Y', strtotime('first day of last month'))
                ]))
            ],
            [
                'label' => $two_months_ago,
                'url' => add_query_arg(array_merge($base_args, [
                    'start_date' => date('01.m.Y', strtotime('first day of -2 month')),
                    'end_date' => date('t.m.Y', strtotime('first day of -2 month'))
                ]))
            ]
        ];
    }

    public function handle_test_sentry_logging(): void
    {
        // Verify nonce
        if (!wp_verify_nonce($_POST['test_sentry_nonce'], 'test_sentry_logging')) {
            wp_die(__('Security check failed', 'dahannes-connector'));
        }

        $debug_info = [];
        
        // Check if Sentry is disabled
        $sentry_disabled = defined('DAHANNES_DISABLE_SENTRY') && DAHANNES_DISABLE_SENTRY;
        $debug_info['sentry_disabled'] = $sentry_disabled;
        
        if ($sentry_disabled) {
            error_log('DaHannes: Test button clicked but DAHANNES_DISABLE_SENTRY is enabled (Sentry disabled)');
        }

        // Force initialization to test
        try {
            // Try to directly use the logger method that should trigger Sentry init
            error_log('DaHannes: About to test logging...');
            
            // Test regular logging first
            DahannesLogger::log('error', 'Test Sentry logging: Error message from debug screen', [
                'test_type' => 'manual_test',
                'severity' => 'error',
                'debug_info' => 'This is a test error for Sentry integration verification',
                'button_test' => true
            ]);

            error_log('DaHannes: Error log completed');
            
            // Also test direct Sentry message sending (which we know works)
            try {
                \DaHannesConnectorDeps\Sentry\captureMessage('Direct Sentry Test: Manual test from debug screen', \DaHannesConnectorDeps\Sentry\Severity::info());
                error_log('DaHannes: Direct Sentry message sent');
            } catch (\Exception $e) {
                error_log('DaHannes: Direct Sentry message failed: ' . $e->getMessage());
            }
            
            // Try to trigger Sentry flush if available
            if (function_exists('\DaHannesConnectorDeps\Sentry\flush')) {
                \DaHannesConnectorDeps\Sentry\flush();
                error_log('DaHannes: Sentry flush completed');
            } else {
                error_log('DaHannes: Sentry flush function not found');
            }
            
            // Also try alternative flush methods
            if (class_exists('\DaHannesConnectorDeps\Sentry\SentrySdk')) {
                try {
                    \DaHannesConnectorDeps\Sentry\SentrySdk::getCurrentHub()->getClient()->flush();
                    error_log('DaHannes: Alternative Sentry flush completed');
                } catch (\Exception $e) {
                    error_log('DaHannes: Alternative flush failed: ' . $e->getMessage());
                }
            }

        } catch (\Exception $e) {
            error_log('DaHannes: Exception during test logging: ' . $e->getMessage());
            $debug_info['exception'] = $e->getMessage();
        }

        // Redirect back with success message and debug info
        wp_redirect(add_query_arg([
            'page' => 'dahannes-status',
            'sentry_test' => 'success',
            'debug' => base64_encode(json_encode($debug_info))
        ], admin_url('admin.php')));
        exit;
    }

    /**
     * Fast atomic reset of sync status using direct database queries
     * 
     * @param array $order_ids Array of order IDs to reset
     * @return int Number of orders reset
     */
    private function atomic_reset_sync_status(array $order_ids): int {
        if (empty($order_ids)) {
            return 0;
        }

        global $wpdb;
        
        // HPOS compatible - check if orders are stored in custom tables
        $is_hpos = wc_get_container()->get(\Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController::class)->custom_orders_table_usage_is_enabled();
        
        if ($is_hpos) {
            // HPOS: Update order meta in the custom orders table
            $meta_table = $wpdb->prefix . 'wc_orders_meta';
            $placeholders = implode(',', array_fill(0, count($order_ids), '%d'));
            
            $updated = $wpdb->query(
                $wpdb->prepare(
                    "UPDATE {$meta_table} 
                     SET meta_value = '0' 
                     WHERE meta_key = %s 
                     AND order_id IN ({$placeholders})",
                    DahannesConnectorUtils::DAHANNES_SUBMISSION_STATUS,
                    ...$order_ids
                )
            );
        } else {
            // Legacy: Update order meta in postmeta table  
            $placeholders = implode(',', array_fill(0, count($order_ids), '%d'));
            
            $updated = $wpdb->query(
                $wpdb->prepare(
                    "UPDATE {$wpdb->postmeta} 
                     SET meta_value = '0' 
                     WHERE meta_key = %s 
                     AND post_id IN ({$placeholders})",
                    DahannesConnectorUtils::DAHANNES_SUBMISSION_STATUS,
                    ...$order_ids
                )
            );
        }
        
        return $updated !== false ? count($order_ids) : 0;
    }
}
