<?php

namespace DaHannesConnector\includes;

use DaHannesConnector\DahannesConnector;
use Exception;
use WC_Order;
use WC_Order_Query;

class DahannesConnectorUtils
{
    const DAHANNES_CODE_SLUG = 'dahannes_member_code';
    const DAHANNES_QUAL_SLUG = 'dahannes_qualifies_for';
    const DAHANNES_SUBMISSION_STATUS = 'dahannes_bonus_submission_status';
    const DAHANNES_PAYLOAD_HASH = 'dahannes_last_payload_hash';
    const SYNC_DELAY_SECONDS = 30;

    /**
     * Set connection failed state by updating is_active to false
     */
    private static function set_connection_failed_state(): void
    {
        $existing_info = get_option('dahannes_info', []);
        
        // Handle case where option is stored as JSON string
        if (is_string($existing_info)) {
            $existing_info = json_decode($existing_info, true) ?: [];
        }
        
        // Ensure we have an array
        if (!is_array($existing_info)) {
            $existing_info = [];
        }
        
        // Preserve existing data but mark as inactive
        if (!empty($existing_info)) {
            $existing_info['is_active'] = false;
            update_option('dahannes_info', $existing_info);
            
            $context = DahannesLogger::enrich_api_context([
                'operation' => 'connection_failure',
                'reason' => get_option('dahannes_api_error', 'Unknown error'),
                'previous_state' => array_intersect_key($existing_info, array_flip(['name', 'app_type', 'data_version', 'is_active'])),
                'timestamp' => current_time('mysql')
            ], get_option('dahannes_server_url'), get_option('api_key'));
            
            DahannesLogger::log('warning', 'Connection failed - marking plugin as inactive', $context);
        }
    }

    public static function update_info_from_dahannes_api(): void
    {
        $client = new ApiClient();

        if (!$client->api_credentials_available()) {
            $context = DahannesLogger::enrich_api_context([
                'operation' => 'update_info_from_api',
                'failure_reason' => 'credentials_unavailable',
                'server_url_configured' => !empty(get_option('dahannes_server_url')),
                'api_key_configured' => !empty(get_option('api_key'))
            ], get_option('dahannes_server_url'), get_option('api_key'));
            
            DahannesLogger::log('error', 'API credentials not available for info update', $context);
            
            update_option('dahannes_api_error', 'API Zugangsdaten fehlen oder sind ungültig');
            update_option('dahannes_api_success', false);
            self::set_connection_failed_state();
            return;
        }

        $start_time = microtime(true);
        $response = $client->get('/api/client/info');
        $response_time_ms = round((microtime(true) - $start_time) * 1000, 2);

        if (isset($response['error'])) {
            $context = DahannesLogger::enrich_api_context([
                'operation' => 'update_info_from_api',
                'endpoint' => '/api/client/info',
                'failure_reason' => 'api_error_response',
                'api_error' => $response['error'],
                'response_time_ms' => $response_time_ms
            ], get_option('dahannes_server_url'), get_option('api_key'));
            
            DahannesLogger::log('error', 'API returned error during info update', $context);
            
            update_option('dahannes_api_error', $response['error']);
            update_option('dahannes_api_success', false);
            self::set_connection_failed_state();
            return;
        }

        if (!isset($response['success']['info'])) {
            $context = DahannesLogger::enrich_api_context([
                'operation' => 'update_info_from_api',
                'endpoint' => '/api/client/info',
                'failure_reason' => 'invalid_response_structure',
                'response_keys' => array_keys($response['success'] ?? []),
                'response_time_ms' => $response_time_ms
            ], get_option('dahannes_server_url'), get_option('api_key'));
            
            DahannesLogger::log('error', 'Invalid API response structure during info update', $context);
            
            update_option('dahannes_api_error', 'Ungültige Antwort vom Server');
            update_option('dahannes_api_success', false);
            self::set_connection_failed_state();
            return;
        }

        // Get the new info data and validate it's an array
        $new_info = $response['success']['info'] ?? null;
        if (!is_array($new_info)) {
            $context = DahannesLogger::enrich_api_context([
                'operation' => 'update_info_from_api',
                'endpoint' => '/api/client/info',
                'failure_reason' => 'invalid_info_data_type',
                'info_data_type' => gettype($new_info),
                'response_time_ms' => $response_time_ms
            ], get_option('dahannes_server_url'), get_option('api_key'));
            
            DahannesLogger::log('error', 'API info data is not an array during info update', $context);
            
            update_option('dahannes_api_error', 'Ungültige Datenstruktur vom Server');
            update_option('dahannes_api_success', false);
            self::set_connection_failed_state();
            return;
        }
        
        // Get the existing info data and ensure it's an array
        $existing_info = get_option('dahannes_info', []);
        if (!is_array($existing_info)) {
            $existing_info = [];
        }
        
        // Check if there's a change in the info data
        if (json_encode($existing_info) !== json_encode($new_info)) {
            $context = DahannesLogger::enrich_api_context([
                'operation' => 'update_info_from_api',
                'endpoint' => '/api/client/info',
                'change_detected' => true,
                'old_info' => is_array($existing_info) ? array_intersect_key($existing_info, array_flip(['name', 'app_type', 'data_version', 'is_active'])) : [],
                'new_info' => is_array($new_info) ? array_intersect_key($new_info, array_flip(['name', 'app_type', 'data_version', 'is_active'])) : [],
                'response_time_ms' => $response_time_ms
            ], get_option('dahannes_server_url'), get_option('api_key'));
            
            DahannesLogger::log('info', 'Connection info updated with changes', $context);
        } else {
            // Log simple message for no change (debug level only)
            $context = DahannesLogger::enrich_api_context([
                'operation' => 'update_info_from_api',
                'endpoint' => '/api/client/info',
                'change_detected' => false,
                'response_time_ms' => $response_time_ms
            ], get_option('dahannes_server_url'), get_option('api_key'));
            
            DahannesLogger::log('debug', 'Connection successful, no changes detected', $context);
        }

        // Handle the success scenario
        update_option('dahannes_info', $new_info);
        delete_option('dahannes_api_error');
        update_option('dahannes_api_success', true);
    }



    public static function compile_list_of_orders_for_server(bool $ignore_delay = false, int $max_orders_per_run = 100): array
    {
        // Get shop type information
        $info_data = get_option('dahannes_info', []);
        $shop_type = $info_data['app_type'] ?? '';
        $data_version = $info_data['data_version'] ?? '';

        // Get orders in two batches: tracked orders needing resubmission + untracked orders with member codes
        $all_orders = [];

        try {
            // First: Get orders with submission_status < data_version (already tracked, need resubmission)
            // Exclude pending/draft orders - only sync orders in a final state
            $tracked_orders_args = [
                'limit' => $max_orders_per_run,
                'status' => ['completed', 'processing', 'cancelled', 'refunded', 'failed', 'on-hold'],
                'meta_query' => [
                    [
                        'key' => self::DAHANNES_SUBMISSION_STATUS,
                        'value' => (int)$data_version,
                        'compare' => '<',
                        'type' => 'NUMERIC'
                    ]
                ],
                'type' => 'shop_order',
            ];
            $tracked_orders = wc_get_orders($tracked_orders_args);

            // Second: Get orders WITHOUT submission_status (untracked orders) - limit remaining slots
            $remaining_slots = max(0, $max_orders_per_run - count($tracked_orders));

            if (count($tracked_orders) >= $max_orders_per_run) {
                DahannesLogger::log('info', 'Order processing limited to prevent long-running process', [
                    'max_orders_per_run' => $max_orders_per_run,
                    'tracked_orders_found' => count($tracked_orders),
                    'remaining_slots' => $remaining_slots,
                    'operation' => 'compile_list_of_orders_for_server'
                ]);
            }

            $untracked_orders_args = [
                'limit' => $remaining_slots,
                'status' => ['completed', 'processing', 'cancelled', 'refunded', 'failed', 'on-hold'],
                'meta_query' => [
                    [
                        'key' => self::DAHANNES_SUBMISSION_STATUS,
                        'compare' => 'NOT EXISTS'
                    ]
                ],
                'type' => 'shop_order',
            ];
            $untracked_orders = wc_get_orders($untracked_orders_args);

            // Combine both sets of orders
            $all_orders = array_merge($tracked_orders, $untracked_orders);

            // Remove duplicates based on order ID
            $unique_orders = [];
            foreach ($all_orders as $order) {
                $unique_orders[$order->get_id()] = $order;
            }
            $orders = array_values($unique_orders);

        } catch (Exception $e) {
            DahannesLogger::log('error', 'Exception querying orders: ' . $e->getMessage());
            return [];
        }


        // for each order, get the order total and member code from meta
        $order_data = [];
        $skipped_order_ids = []; // Collect orders to mark as processed in batch

        foreach ($orders as $order) {

            // Skip refunds - check the order type directly
            if ($order->get_type() !== 'shop_order') {
                $skipped_order_ids[] = $order->get_id();
                continue;
            }

            // Check if order creation delay has expired (using WooCommerce's built-in timestamp)
            if (!$ignore_delay) {
                $created_at = $order->get_date_created();
                $current_time = time();

                if ($created_at) {
                    $created_timestamp = $created_at->getTimestamp();
                    $age_seconds = $current_time - $created_timestamp;

                    if ($age_seconds < self::SYNC_DELAY_SECONDS) {
                        $remaining = self::SYNC_DELAY_SECONDS - $age_seconds;
                        DahannesLogger::log('debug', 'Order still in waiting period', [
                            'order_id' => $order->get_id(),
                            'age_seconds' => $age_seconds,
                            'remaining_seconds' => $remaining,
                            'operation' => 'compile_list_of_orders_for_server'
                        ]);
                        $skipped_order_ids[] = $order->get_id();
                        continue;
                    }
                }
            }

            $code = self::get_member_code_from_order($order);

            // For partner shops: skip orders without a code or with invalid code length
            if ($shop_type === 'partner' && (empty($code) || strlen($code) < 5 || strlen($code) > 7)) {
                $skipped_order_ids[] = $order->get_id();
                continue;
            }

            // For non-partner shops, include orders even if code is empty
            $qualifies_for = $order->get_meta(self::DAHANNES_QUAL_SLUG, true);

            // Calculate subtotal minus discounts, refunds, and products with disabled bonus
            /** @var WC_Order $order */
            $order_total = $order->get_total();
            $percentage_refunded = $order_total > 0 ? $order->get_total_refunded() / $order_total : 0;

            // Calculate subtotal only for products that don't have bonus disabled
            $subtotal = 0;
            foreach ($order->get_items() as $item) {
                $product_id = $item->get_product_id();

                // Skip products with disabled bonus - HPOS compatible
                $product = wc_get_product($product_id);

                $disable_bonus = $product ? $product->get_meta('_dahannes_disable_bonus', true) : '';
                if ($disable_bonus === 'yes') {
                    continue;
                }

                // Add item subtotal to order subtotal
                $subtotal += $item->get_subtotal();
            }

            $subtotal_not_refunded = $subtotal * (1 - $percentage_refunded);

            // Determine if order is active (based on status only)
            $is_active = in_array($order->get_status(), ['completed', 'processing']);

            $order_data[$order->get_id()] = [
                'order_id' => $order->get_id(),
                'order_date' => $order->get_date_created()->format('Y-m-d H:i:s'),
                'order_total' => $subtotal_not_refunded,
                'order_qualifies_for' => $qualifies_for,
                'member_code' => $code, // Keep code as is, even if empty
                'active' => $is_active,
            ];
        }

        // Batch update skipped orders to mark them as processed
        if (!empty($skipped_order_ids)) {
            self::mark_orders_processed($skipped_order_ids, $info_data['data_version']);

            DahannesLogger::log('info', 'Marked skipped orders as processed', [
                'skipped_count' => count($skipped_order_ids),
                'operation' => 'compile_list_of_orders_for_server'
            ]);
        }

        return $order_data;
    }

    /**
     * Calculate hash of mutable order data that gets sent to server
     *
     * @param WC_Order|int $order Order object or order ID
     * @param array $product_cache Optional pre-loaded product cache to avoid repeated queries
     * @return string MD5 hash of mutable order data
     */
    public static function calculate_order_payload_hash($order, array $product_cache = []): string {
        if (is_numeric($order)) {
            $order = wc_get_order($order);
        }

        if (!$order) {
            return '';
        }

        // Get shop type and calculate the same data that goes to server
        $info_data = get_option('dahannes_info', []);
        $shop_type = $info_data['app_type'] ?? '';

        $code = self::get_member_code_from_order($order);

        // Skip orders without member code for partner shops (same logic as compile_list_of_orders_for_server)
        if ($shop_type === 'partner' && (empty($code) || strlen($code) < 5 || strlen($code) > 7)) {
            return '';
        }

        $qualifies_for = $order->get_meta(self::DAHANNES_QUAL_SLUG, true);

        // Calculate order total the same way as in compile_list_of_orders_for_server
        $order_total = $order->get_total();
        $percentage_refunded = $order_total > 0 ? $order->get_total_refunded() / $order_total : 0;

        $subtotal = 0;
        foreach ($order->get_items() as $item) {
            $product_id = $item->get_product_id();

            // Use cached product if available, otherwise load it
            $product = $product_cache[$product_id] ?? wc_get_product($product_id);
            $disable_bonus = $product ? $product->get_meta('_dahannes_disable_bonus', true) : '';
            if ($disable_bonus === 'yes') {
                continue;
            }

            $subtotal += $item->get_subtotal();
        }

        $subtotal_not_refunded = $subtotal * (1 - $percentage_refunded);
        $is_active = in_array($order->get_status(), ['completed', 'processing']);

        // Create hash of only the mutable fields that are sent to server
        $mutable_data = [
            'order_total' => $subtotal_not_refunded,
            'order_qualifies_for' => $qualifies_for,
            'member_code' => $code,
            'active' => $is_active,
        ];

        return md5(serialize($mutable_data));
    }

    /**
     * Mark orders as processed with the current data version
     *
     * Uses atomic SQL operations for performance.
     * Returns detailed results about successes and failures.
     * Failed orders will be retried on the next cron run.
     *
     * @param array $order_ids Array of order IDs to mark as processed
     * @param string $data_version Current data version from API
     * @return array ['success' => int, 'failed' => array, 'errors' => array]
     */
    public static function mark_orders_processed(array $order_ids, string $data_version): array {
        if (empty($order_ids)) {
            return ['success' => 0, 'failed' => [], 'errors' => []];
        }

        global $wpdb;

        // Check if using HPOS or legacy postmeta
        $is_hpos = wc_get_container()->get(
            \Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController::class
        )->custom_orders_table_usage_is_enabled();

        // Atomic update for submission status
        $placeholders = implode(',', array_fill(0, count($order_ids), '%d'));

        if ($is_hpos) {
            $status_updated = $wpdb->query(
                $wpdb->prepare(
                    "UPDATE {$wpdb->prefix}wc_orders_meta
                     SET meta_value = %s
                     WHERE meta_key = %s
                     AND order_id IN ({$placeholders})",
                    $data_version,
                    self::DAHANNES_SUBMISSION_STATUS,
                    ...$order_ids
                )
            );
        } else {
            $status_updated = $wpdb->query(
                $wpdb->prepare(
                    "UPDATE {$wpdb->postmeta}
                     SET meta_value = %s
                     WHERE meta_key = %s
                     AND post_id IN ({$placeholders})",
                    $data_version,
                    self::DAHANNES_SUBMISSION_STATUS,
                    ...$order_ids
                )
            );
        }

        // Batch load orders for hash calculation
        $orders = wc_get_orders([
            'include' => $order_ids,
            'limit' => count($order_ids),  // Explicitly limit to prevent WooCommerce from loading all orders
            'return' => 'objects'
        ]);

        $hash_values = [];

        // Pre-load all unique products to avoid repeated database queries
        $product_ids = [];
        foreach ($orders as $order) {
            foreach ($order->get_items() as $item) {
                $product_ids[] = $item->get_product_id();
            }
        }
        $product_ids = array_unique($product_ids);

        // Load all products at once and cache them
        $product_cache = [];
        if (!empty($product_ids)) {
            $products = wc_get_products(['include' => $product_ids, 'limit' => -1]);
            foreach ($products as $product) {
                $product_cache[$product->get_id()] = $product;
            }
        }

        // Calculate hashes for each order
        foreach ($orders as $order) {
            $order_id = $order->get_id();
            $hash = self::calculate_order_payload_hash($order, $product_cache);

            if (!empty($hash)) {
                $hash_values[] = [
                    'order_id' => $order_id,
                    'hash' => $hash
                ];
            }
        }

        // Atomic bulk hash insert/update using multi-row INSERT
        if (!empty($hash_values)) {
            $values_sql = [];
            $prepare_args = [];

            foreach ($hash_values as $item) {
                $values_sql[] = '(%d, %s, %s)';
                $prepare_args[] = $item['order_id'];
                $prepare_args[] = self::DAHANNES_PAYLOAD_HASH;
                $prepare_args[] = $item['hash'];
            }

            $values_string = implode(', ', $values_sql);

            if ($is_hpos) {
                $wpdb->query(
                    $wpdb->prepare(
                        "INSERT INTO {$wpdb->prefix}wc_orders_meta (order_id, meta_key, meta_value)
                         VALUES {$values_string}
                         ON DUPLICATE KEY UPDATE meta_value = VALUES(meta_value)",
                        ...$prepare_args
                    )
                );
            } else {
                $wpdb->query(
                    $wpdb->prepare(
                        "INSERT INTO {$wpdb->postmeta} (post_id, meta_key, meta_value)
                         VALUES {$values_string}
                         ON DUPLICATE KEY UPDATE meta_value = VALUES(meta_value)",
                        ...$prepare_args
                    )
                );
            }
        }

        // Insert status meta for any orders that didn't have it (new orders)
        $orders_needing_insert = count($order_ids) - $status_updated;
        if ($orders_needing_insert > 0) {
            foreach ($order_ids as $order_id) {
                if ($is_hpos) {
                    $wpdb->query(
                        $wpdb->prepare(
                            "INSERT IGNORE INTO {$wpdb->prefix}wc_orders_meta (order_id, meta_key, meta_value)
                             VALUES (%d, %s, %s)",
                            $order_id,
                            self::DAHANNES_SUBMISSION_STATUS,
                            $data_version
                        )
                    );
                } else {
                    $wpdb->query(
                        $wpdb->prepare(
                            "INSERT IGNORE INTO {$wpdb->postmeta} (post_id, meta_key, meta_value)
                             VALUES (%d, %s, %s)",
                            $order_id,
                            self::DAHANNES_SUBMISSION_STATUS,
                            $data_version
                        )
                    );
                }
            }
        }

        return [
            'success' => count($order_ids),
            'failed' => [],
            'errors' => []
        ];
    }

    public static function save_dh_value_to_cookie_and_user_meta(): void
    {
        // 1. Check if 'dh' GET parameter is set and save to cookie
        $param = $_GET['sponsor'] ?? $_GET['dh'] ?? null;
        if ($param) {
            // Sanitize the param value.
            $dh_value = sanitize_text_field($param);

            // Set cookie to expire in 1 year. Replace existing cookie if it exists.
            setcookie(static::DAHANNES_CODE_SLUG, $dh_value, time() + (365 * 24 * 60 * 60), '/');
        }

        // 2. If user is logged in and doesn't have the user_meta set, check cookie and set user_meta
        if (is_user_logged_in()) {
            $user_id = get_current_user_id();

            $existing_meta = get_user_meta($user_id, static::DAHANNES_CODE_SLUG, true);

            if (empty($existing_meta) && isset($_COOKIE[static::DAHANNES_CODE_SLUG])) {
                $cookie_value = sanitize_text_field($_COOKIE[static::DAHANNES_CODE_SLUG]);
                update_user_meta($user_id, static::DAHANNES_CODE_SLUG, $cookie_value);
            }
        }
    }

    /**
     * Register additional checkout field using WooCommerce 8.9+ Additional Checkout Fields API
     * This provides compatibility with both classic and block checkout
     */
    public static function register_additional_checkout_field(): void
    {
        add_action('woocommerce_init', function () {
            // Check if the function exists (WooCommerce 8.9+)
            if (!function_exists('woocommerce_register_additional_checkout_field')) {
                return; // Fallback to classic implementation
            }

            $dahannes_info = get_option('dahannes_info', []);
            $shop_type = $dahannes_info['app_type'] ?? '';
            $is_required = ($shop_type !== 'partner');
            
            // No automatic prefilling - users must manually enter their code

            woocommerce_register_additional_checkout_field([
                'id'       => 'dahannes-connector/member-code',
                'label'    => __('Dahannes Mitgliedscode', 'dahannes-connector'),
                'location' => 'order',
                'type'     => 'text',
                'required' => $is_required,
                'sanitize_callback' => function ($value) {
                    return trim(strtolower(sanitize_text_field($value)));
                },
                'validate_callback' => function ($value) {
                    $dahannes_info = get_option('dahannes_info', []);
                    $shop_type = $dahannes_info['app_type'] ?? '';
                    
                    // Empty value handling
                    if (empty($value)) {
                        if ($shop_type !== 'partner') {
                            return new \WP_Error(
                                'missing_member_code',
                                __('Bitte geben Sie einen Dahannes Mitgliedscode ein.', 'dahannes-connector')
                            );
                        }
                        return true; // Valid for partner shops
                    }
                    
                    // Format validation if code is provided
                    if ($value !== 'x' && !preg_match('/^[A-Za-z0-9]{5,7}$/', $value)) {
                        $message = $shop_type === 'partner'
                            ? __('Der Mitgliedscode muss 5-7 Zeichen lang sein.', 'dahannes-connector')
                            : __('Der Mitgliedscode muss 5-7 Zeichen lang sein oder ein „x" sein.', 'dahannes-connector');
                        return new \WP_Error('invalid_member_code', $message);
                    }
                    
                    return true;
                }
            ]);
            
            // Provide default values for the field using WooCommerce filter (exclude "X")
            add_filter('woocommerce_get_default_value_for_dahannes-connector/member-code', function ($value, $group, $wc_object) {
                // Only provide default if no value is set and user has a real saved code (not "X")
                if (empty($value)) {
                    $saved_code = self::get_member_code();
                    // Only prefill real member codes, not "X"
                    if (!empty($saved_code) && strtolower($saved_code) !== 'x') {
                        return $saved_code;
                    }
                }
                return $value;
            }, 10, 3);

            // Add JavaScript for help text only (no prefilling)
            add_action('wp_footer', function () use ($shop_type) {
                // Only add on checkout page
                if (!is_checkout() || is_order_received_page()) {
                    return;
                }
                ?>
                <script type="text/javascript">
                    jQuery(document).ready(function($) {
                        let dahannesFieldProcessed = false;
                        
                        // Function to handle field setup
                        function setupDahannesField() {
                            // Prevent multiple executions
                            if (dahannesFieldProcessed) {
                                return;
                            }
                            
                            // Look for member code field in block checkout
                            const memberCodeField = $('input[name*="member-code"], input[id*="member-code"], input[data-field-key*="member-code"]');
                            
                            if (memberCodeField.length > 0) {
                                dahannesFieldProcessed = true;
                                
                                // Add help text for non-partner shops (only once)
                                <?php if ($shop_type !== 'partner'): ?>
                                // Remove any existing help text first
                                $('.dahannes-help-text').remove();
                                
                                const helpText = '<?php echo esc_js(__('Sollten Sie keinen Code haben, geben Sie ein „X" ein.', 'dahannes-connector')); ?>';
                                const fieldContainer = memberCodeField.closest('.wp-block-group, .wc-block-components-text-input, .form-row');
                                
                                if (fieldContainer.length > 0 && fieldContainer.siblings('.dahannes-help-text').length === 0) {
                                    fieldContainer.after('<p class="dahannes-help-text" style="color: #666; font-size: 0.85em; margin-top: 0.5em;">' + helpText + '</p>');
                                }
                                <?php endif; ?>
                            }
                        }
                        
                        // Reset flag when checkout updates to allow re-processing
                        $(document).on('updated_checkout', function() {
                            dahannesFieldProcessed = false;
                            setTimeout(setupDahannesField, 200);
                        });
                        
                        // Run initially with a small delay to ensure DOM is ready
                        setTimeout(setupDahannesField, 500);
                        
                        // For block checkout, watch for new elements but with throttling
                        let observerTimeout;
                        const observer = new MutationObserver(function(mutations) {
                            // Clear existing timeout
                            if (observerTimeout) {
                                clearTimeout(observerTimeout);
                            }
                            
                            // Set new timeout to prevent excessive calls
                            observerTimeout = setTimeout(function() {
                                dahannesFieldProcessed = false;
                                setupDahannesField();
                            }, 300);
                        });
                        
                        const checkoutContainer = document.querySelector('.wc-block-checkout, .checkout');
                        if (checkoutContainer) {
                            observer.observe(checkoutContainer, { 
                                childList: true, 
                                subtree: true 
                            });
                        }
                    });
                </script>
                <?php
            });
        });
    }

    public static function add_dh_checkout_field($checkout): void
    {
        // Get saved value but exclude "X" from prefilling
        $saved_value = self::get_member_code();
        $prefill_value = '';
        
        // Only prefill if it's a real member code (not "X")
        if (!empty($saved_value) && strtolower($saved_value) !== 'x') {
            $prefill_value = $saved_value;
        }
        
        $dahannes_info = get_option('dahannes_info', []);
        $shop_type = $dahannes_info['app_type'] ?? '';
        $is_required = ($shop_type !== 'partner');

        woocommerce_form_field('dh_checkout_field', array(
            'type' => 'text',
            'class' => array('form-row-wide' . ($is_required ? ' validate-required' : '')),
            'label' => __('Dahannes Mitgliedscode', 'dahannes-connector'),
            'required' => $is_required,
            'default' => $prefill_value,
        ), $checkout->get_value('dh_checkout_field') ?: $prefill_value);

        // Add help text based on shop type
        if ($shop_type !== 'partner') {
            echo '<p class="form-row form-row-wide" style="margin-top: 0.5em; font-size: 0.85em;">Sollten Sie keinen Code haben, geben Sie ein „X" ein.</p>';
        }

        // Add client-side validation
        ?>
        <script type="text/javascript">
            jQuery(function ($) {
                const isPartner = <?php echo $shop_type === 'partner' ? 'true' : 'false'; ?>;

                function validateDahannesCode(code) {
                    code = code.trim();
                    if (isPartner) {
                        return code === '' || /^[A-Za-z0-9]{5,7}$/.test(code);
                    }
                    return code.toLowerCase() === 'x' || /^[A-Za-z0-9]{5,7}$/.test(code);
                }

                // Real-time validation
                $(document).on('input change', '#dh_checkout_field', function () {
                    const $field = $(this);
                    const $row = $field.closest('.form-row');

                    // Remove existing validation classes
                    $row.removeClass('woocommerce-validated woocommerce-invalid');

                    // Empty is valid for partner shops
                    if ($field.val().trim() === '' && isPartner) {
                        $row.addClass('woocommerce-validated');
                        return;
                    }

                    // Add appropriate class based on validation
                    if (validateDahannesCode($field.val())) {
                        $row.addClass('woocommerce-validated');
                    } else {
                        $row.addClass('woocommerce-invalid');
                    }
                });

                // Form submission validation
                $('form.checkout').on('checkout_place_order', function () {
                    const $field = $('#dh_checkout_field');
                    const value = $field.val().trim();

                    // Empty is valid for partner shops
                    if (value === '' && isPartner) {
                        return true;
                    }

                    const isValid = validateDahannesCode(value);

                    if (!isValid) {
                        const $row = $field.closest('.form-row');
                        $row.removeClass('woocommerce-validated').addClass('woocommerce-invalid');

                        if ($('.woocommerce-error').length === 0) {
                            const errorMessage = isPartner
                                ? 'Der Mitgliedscode muss 5-7 Zeichen lang sein.'
                                : 'Der Mitgliedscode muss 5-7 Zeichen lang sein oder ein „x" sein.';
                            $('<ul class="woocommerce-error"><li>' + errorMessage + '</li></ul>')
                                .prependTo('form.checkout');
                        }
                    }

                    return isValid;
                });
            });
        </script>
        <?php
    }


    public static function save_dh_checkout_field($order_id): void
    {
        $field_value = self::get_member_code_from_checkout_data();
        
        if (!empty($field_value)) {
            // Sanitize and format the code consistently
            $field_value = trim(strtolower(sanitize_text_field($field_value)));
            
            // HPOS compatible - use order object
            $order = wc_get_order($order_id);
            if ($order) {
                $order->update_meta_data(static::DAHANNES_CODE_SLUG, $field_value);
                
                // Initialize submission status if not set
                if (!$order->get_meta(self::DAHANNES_SUBMISSION_STATUS, true)) {
                    $order->update_meta_data(self::DAHANNES_SUBMISSION_STATUS, '0');
                }
                
                // Prevent hook recursion during save
                remove_action('woocommerce_update_order', [self::class, 'hook_update_order']);
                $order->save();
                add_action('woocommerce_update_order', [self::class, 'hook_update_order']);
            }

            // Also update user meta for logged-in users
            $user_id = get_current_user_id();
            if ($user_id) {
                update_user_meta($user_id, static::DAHANNES_CODE_SLUG, $field_value);
            }
        }
        
        // Also check for product qualifications after saving the checkout field
        // This ensures qualifications are set even if save_post_shop_order was called before checkout fields were saved
        static::check_member_code_and_product_meta($order_id);
    }
    
    /**
     * Get member code from checkout data - supports both classic and block checkout
     */
    private static function get_member_code_from_checkout_data(): string
    {
        // Try classic checkout field first
        if (!empty($_POST['dh_checkout_field'])) {
            return sanitize_text_field($_POST['dh_checkout_field']);
        }
        
        // Try additional checkout fields API (WooCommerce 8.9+)
        if (function_exists('woocommerce_register_additional_checkout_field')) {
            // The additional checkout fields are stored in a different part of $_POST
            // They typically come through the Store API in a structured format
            
            // Check if we're dealing with a block checkout submission
            if (isset($_POST['extensions']) && is_array($_POST['extensions'])) {
                // Look for our field in the extensions data
                foreach ($_POST['extensions'] as $extension_data) {
                    if (is_array($extension_data) && isset($extension_data['dahannes-connector/member-code'])) {
                        return sanitize_text_field($extension_data['dahannes-connector/member-code']);
                    }
                }
            }
            
            // Also check direct field access pattern for additional checkout fields
            $field_patterns = [
                'dahannes-connector_member-code',
                'dahannes-connector-member-code',
                'member-code',
                'dahannes_member_code'
            ];
            
            foreach ($field_patterns as $pattern) {
                if (!empty($_POST[$pattern])) {
                    return sanitize_text_field($_POST[$pattern]);
                }
            }
        }
        
        return '';
    }
    
    /**
     * Handle block checkout field saving from Store API
     * This is called by woocommerce_store_api_checkout_update_order_from_request
     */
    public static function save_block_checkout_field(\WC_Order $order, \WP_REST_Request $request): void
    {
        // The Additional Checkout Fields API automatically handles saving
        // but we need to ensure the data is also saved to our classic meta field for compatibility
        try {
            if (function_exists('woocommerce_register_additional_checkout_field')) {
                // Check if WooCommerce Blocks classes are available
                if (class_exists('Automattic\WooCommerce\Blocks\Package') && 
                    class_exists('Automattic\WooCommerce\Blocks\Domain\Services\CheckoutFields')) {
                    
                    $checkout_fields = \Automattic\WooCommerce\Blocks\Package::container()
                        ->get(\Automattic\WooCommerce\Blocks\Domain\Services\CheckoutFields::class);
                    
                    $field_value = $checkout_fields->get_field_from_object(
                        'dahannes-connector/member-code',
                        $order,
                        'other'
                    );
                    
                    if (!empty($field_value)) {
                        // Sanitize and format consistently
                        $field_value = trim(strtolower(sanitize_text_field($field_value)));
                        
                        // Save to classic meta for backward compatibility - HPOS compatible
                        $order->update_meta_data(self::DAHANNES_CODE_SLUG, $field_value);
                        
                        // Initialize submission status if not set
                        if (!$order->get_meta(self::DAHANNES_SUBMISSION_STATUS, true)) {
                            $order->update_meta_data(self::DAHANNES_SUBMISSION_STATUS, '0');
                        }
                        
                        // Prevent hook recursion during save
                        remove_action('woocommerce_update_order', [self::class, 'hook_update_order']);
                        $order->save();
                        add_action('woocommerce_update_order', [self::class, 'hook_update_order']);
                        
                        // Also update user meta for logged-in users
                        $user_id = $order->get_user_id();
                        if ($user_id) {
                            update_user_meta($user_id, self::DAHANNES_CODE_SLUG, $field_value);
                        }
                    }
                }
            }
        } catch (\Exception $e) {
            DahannesLogger::log('warning', 'Failed to save block checkout field', [
                'order_id' => $order->get_id(),
                'error' => $e->getMessage()
            ]);
        }
        
        // Ensure product qualifications are processed
        self::check_member_code_and_product_meta($order->get_id());
    }

    public static function hook_save_post_shop_order($order_id): void
    {
        // Check if this is a shop_order and not a shop_order_refund
        if (get_post_type($order_id) !== 'shop_order') {
            return;
        }

        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }

        // Get the old and new status
        $old_status = $order->get_status();
        $new_status = $_POST['order_status'] ?? '';

        // Helper function to check if status is completed or processing
        $is_relevant_status = function($status) {
            return in_array($status, ['completed', 'processing']);
        };

        $needs_save = false;

        // If status changed TO or FROM completed/processing
        if ($is_relevant_status($old_status) !== $is_relevant_status($new_status)) {
            // Set submission status to 0 to trigger re-sync - HPOS compatible
            $order->update_meta_data(self::DAHANNES_SUBMISSION_STATUS, '0');
            $needs_save = true;
        }
        // For new orders, always set initial submission status
        elseif (!$order->get_meta(self::DAHANNES_SUBMISSION_STATUS, true)) {
            $order->update_meta_data(self::DAHANNES_SUBMISSION_STATUS, '0');
            $needs_save = true;
        }
        
        // Save only if needed and prevent recursion
        if ($needs_save) {
            remove_action('woocommerce_update_order', [self::class, 'hook_update_order']);
            $order->save();
            add_action('woocommerce_update_order', [self::class, 'hook_update_order']);
        }
        
        static::check_member_code_and_product_meta($order_id);
    }

    /**
     * Handle new HPOS orders
     */
    public static function hook_new_order(int $order_id): void
    {
        $order = wc_get_order($order_id);
        if (!$order || $order->get_type() !== 'shop_order') {
            return;
        }

        // Initialize submission status for new orders
        if (!$order->get_meta(self::DAHANNES_SUBMISSION_STATUS, true)) {
            $order->update_meta_data(self::DAHANNES_SUBMISSION_STATUS, '0');

            // Temporarily remove update hook to prevent recursion
            remove_action('woocommerce_update_order', [self::class, 'hook_update_order']);
            $order->save();
            add_action('woocommerce_update_order', [self::class, 'hook_update_order']);
        }

        static::check_member_code_and_product_meta($order_id);
    }

    /**
     * Handle HPOS order updates
     */
    public static function hook_update_order(int $order_id): void
    {
        // Prevent infinite recursion
        static $processing_orders = [];
        if (isset($processing_orders[$order_id])) {
            return;
        }
        $processing_orders[$order_id] = true;

        $order = wc_get_order($order_id);
        if (!$order || $order->get_type() !== 'shop_order') {
            unset($processing_orders[$order_id]);
            return;
        }

        $needs_save = false;

        // Initialize submission status if it doesn't exist
        $submission_status = $order->get_meta(self::DAHANNES_SUBMISSION_STATUS, true);
        if (!$submission_status) {
            $order->update_meta_data(self::DAHANNES_SUBMISSION_STATUS, '0');
            $needs_save = true;
        }
        
        // Check if order payload has changed since last submission
        $current_hash = self::calculate_order_payload_hash($order);
        $stored_hash = $order->get_meta(self::DAHANNES_PAYLOAD_HASH, true);
        
        // If hash is different (or missing), reset submission status
        if (!empty($current_hash) && $current_hash !== $stored_hash) {
            $order->update_meta_data(self::DAHANNES_SUBMISSION_STATUS, '0');
            $needs_save = true;
        }
        
        // Also check data version in case of server upgrades
        if ($submission_status !== '0') {
            $info_data = get_option('dahannes_info', []);
            $data_version = $info_data['data_version'] ?? '';
            if ($submission_status < $data_version) {
                $order->update_meta_data(self::DAHANNES_SUBMISSION_STATUS, '0');
                $needs_save = true;
            }
        }

        // Save only once if needed
        if ($needs_save) {
            // Temporarily remove the hook to prevent recursion during save
            remove_action('woocommerce_update_order', [self::class, 'hook_update_order']);
            $order->save();
            add_action('woocommerce_update_order', [self::class, 'hook_update_order']);
        }

        static::check_member_code_and_product_meta($order_id);
        
        unset($processing_orders[$order_id]);
    }

    /**
     * React to order status changes from gateways/automation (not just admin saves)
     */
    public static function hook_order_status_changed($order_id, $old_status, $new_status, $order = null): void
    {
        // Normalize order object
        if (!$order instanceof \WC_Order) {
            $order = wc_get_order($order_id);
        }
        if (!$order || $order->get_type() !== 'shop_order') {
            return;
        }

        // Determine if active flag changed
        $is_active = function($status) {
            return in_array($status, ['completed', 'processing']);
        };

        $needs_save = false;

        // If the active state toggled, mark for resubmission
        if ($is_active($old_status) !== $is_active($new_status)) {
            $order->update_meta_data(self::DAHANNES_SUBMISSION_STATUS, '0');
            $needs_save = true;
        }

        // Also compare payload hash; if changed, mark for resubmission
        $current_hash = self::calculate_order_payload_hash($order);
        $stored_hash = $order->get_meta(self::DAHANNES_PAYLOAD_HASH, true);
        if (!empty($current_hash) && $current_hash !== $stored_hash) {
            $order->update_meta_data(self::DAHANNES_SUBMISSION_STATUS, '0');
            $needs_save = true;
        }

        if ($needs_save) {
            // Prevent recursion loops with update hook
            remove_action('woocommerce_update_order', [self::class, 'hook_update_order']);
            $order->save();
            add_action('woocommerce_update_order', [self::class, 'hook_update_order']);
        }

        // Keep qualifications in sync
        self::check_member_code_and_product_meta($order_id);
    }

    public static function check_member_code_and_product_meta($order_id): void
    {
        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }
        
        // Check if sponsor_code is set in order meta
        $member_code = self::get_member_code_from_order($order_id);
        
        $dahannes_info = get_option('dahannes_info', []);
        $shop_type = $dahannes_info['app_type'] ?? '';
        
        if (empty($member_code) && $shop_type === 'partner') {
            return;
        }

        // Store qualification info if present
        $items = $order->get_items();
        $qualifies_for = [];
        
        foreach ($items as $item) {
            $product_id = $item->get_product_id();
            
            // Get qualification regardless of bonus status - HPOS compatible
            $product = wc_get_product($product_id);
            $qualification = $product ? $product->get_meta('_dahannes_gives_qualification', true) : '';
            
            if (!empty($qualification)) {
                $qualifies_for[] = $qualification;
            }
        }
        
        if (!empty($qualifies_for)) {
            $qualifies_for_string = implode(',', array_unique($qualifies_for));
            $existing_qualifies_for = $order->get_meta(self::DAHANNES_QUAL_SLUG, true);
            
            // Only save if the qualification data actually changed
            if ($existing_qualifies_for !== $qualifies_for_string) {
                $order->update_meta_data(self::DAHANNES_QUAL_SLUG, $qualifies_for_string);
                
                // Use a direct update to avoid triggering woocommerce_update_order hook
                remove_action('woocommerce_update_order', [self::class, 'hook_update_order']);
                $order->save();
                add_action('woocommerce_update_order', [self::class, 'hook_update_order']);
            }
        }
    }

    /**
     * Get member code from user meta and cookies (for forms/checkout display)
     * @return mixed|string
     */
    public static function get_member_code(): mixed
    {
        $user_id = get_current_user_id();
        $code = ''; // default value

        // Check user meta first.
        if ($user_id) {
            $code = get_user_meta($user_id, self::DAHANNES_CODE_SLUG, true);
        }

        // If user meta doesn't exist, then check the cookie.
        if (empty($code) && isset($_COOKIE[self::DAHANNES_CODE_SLUG])) {
            $code = sanitize_text_field($_COOKIE[self::DAHANNES_CODE_SLUG]);
        }
        return $code;
    }
    
    /**
     * Get order meta with HPOS-first approach and legacy fallback
     *
     * @param int|WC_Order $order Order ID or WC_Order object
     * @param string $key Meta key to retrieve
     * @param bool $single Whether to return a single value
     * @return mixed Meta value, empty string if not found
     */
    private static function get_order_meta_hpos_safe($order, string $key, bool $single = true)
    {
        // Get order object if ID passed
        if (is_numeric($order)) {
            $order_obj = wc_get_order($order);
            $order_id = $order;
        } else {
            $order_obj = $order;
            $order_id = $order->get_id();
        }

        if (!$order_obj) {
            return $single ? '' : [];
        }

        // Try HPOS first
        $value = $order_obj->get_meta($key, $single);
        if (!empty($value)) {
            return $value;
        }

        // Fallback to legacy post meta for backward compatibility
        return get_post_meta($order_id, $key, $single);
    }

    /**
     * Get member code from order - supports both classic meta and Additional Checkout Fields API
     * @param int|WC_Order $order Order ID or WC_Order object
     * @return string
     */
    public static function get_member_code_from_order($order): string
    {
        if (is_numeric($order)) {
            $order_id = $order;
            $order_obj = wc_get_order($order);
        } else {
            $order_obj = $order;
            $order_id = $order->get_id();
        }
        
        if (!$order_obj) {
            return '';
        }
        
        // First try HPOS-safe meta retrieval (HPOS first, legacy fallback)
        $code = self::get_order_meta_hpos_safe($order_obj, self::DAHANNES_CODE_SLUG, true);
        
        // If not found and Additional Checkout Fields API is available, try that
        if (empty($code) && function_exists('woocommerce_register_additional_checkout_field')) {
            try {
                // Check if WooCommerce Blocks classes are available
                if (class_exists('Automattic\WooCommerce\Blocks\Package') && 
                    class_exists('Automattic\WooCommerce\Blocks\Domain\Services\CheckoutFields')) {
                    
                    $checkout_fields = \Automattic\WooCommerce\Blocks\Package::container()
                        ->get(\Automattic\WooCommerce\Blocks\Domain\Services\CheckoutFields::class);
                    
                    $field_value = $checkout_fields->get_field_from_object(
                        'dahannes-connector/member-code',
                        $order_obj,
                        'other' // 'order' location fields are stored in 'other' group
                    );
                    
                    if (!empty($field_value)) {
                        $code = $field_value;
                    }
                }
            } catch (\Exception $e) {
                // Fallback to classic method if anything goes wrong
                DahannesLogger::log('warning', 'Failed to retrieve member code using Additional Checkout Fields API', [
                    'order_id' => $order_id,
                    'error' => $e->getMessage()
                ]);
            }
        }
        
        return $code ?: '';
    }


    /**
     * Register the DaHannes custom product tab
     */
    public static function register_product_tabs($tabs): array
    {
        // Get shop type to determine if we should show the tab
        $info_data = get_option('dahannes_info', '');
        if (empty($info_data)) {
            return $tabs;
        }

        // Only show tab for commercial and association shops
        $shop_type = strtolower($info_data['app_type'] ?? '');
        if ($shop_type === 'commercial' || $shop_type === 'association') {
            
            $tabs['dahannes'] = array(
                'label'    => __('DaHannes', 'dahannes-connector'),
                'target'   => 'dahannes_product_data',
                'class'    => array('show_if_simple', 'show_if_variable', 'show_if_grouped', 'show_if_external', 'dahannes-icon'),
                'priority' => 90,
            );
        }
        
        return $tabs;
    }

    /**
     * Add the DaHannes product data panel content
     */
    public static function add_product_tab_content(): void
    {
        // Get shop type to determine what fields to show
        $info_data = get_option('dahannes_info', '');
        if (empty($info_data)) {
            return;
        }

        // Only show content for commercial and association shops
        $shop_type = strtolower($info_data['app_type'] ?? '');
        if ($shop_type === 'commercial' || $shop_type === 'association') {
            global $post;
            $product_id = $post->ID;
            $current_value = get_post_meta($product_id, '_dahannes_gives_qualification', true);
            $disable_bonus = get_post_meta($product_id, '_dahannes_disable_bonus', true);
            ?>
            <div id="dahannes_product_data" class="panel woocommerce_options_panel">
                <div class="options_group">
                    <p class="form-field">
                        <strong><?php _e('DaHannes Bonus-Einstellungen', 'dahannes-connector'); ?></strong>
                    </p>
                    
                    <?php
                    // For commercial shops, show qualification dropdown
                    if ($shop_type === 'commercial') {
                        // Define standard options
                        $options = [
                            '' => 'Keine',
                            'welcome-package' => 'Start-Up',
                        ];

                        // Check if this product has a legacy value
                        $has_legacy_value = ($current_value === 'best-of-all');

                        // If it has a legacy value, add it to the options with a "(Legacy)" label
                        if ($has_legacy_value) {
                            $options['best-of-all'] = 'Best of All (Legacy - wird nicht mehr unterstützt)';

                        }

                        woocommerce_wp_select(
                            array(
                                'id' => '_dahannes_gives_qualification',
                                'label' => __('Produkt qualifiziert für', 'woocommerce'),
                                'options' => $options,
                                'value' => $current_value,
                                'description' => $has_legacy_value ?
                                    '<span style="color: #d63638;">Dieser Wert wird nicht mehr unterstützt. Bitte wählen Sie einen neuen Wert.</span>' : '',
                            )
                        );
                    }
                    
                    // Add the toggle field to disable bonus - available for both commercial and association shops
                    woocommerce_wp_checkbox(
                        array(
                            'id' => '_dahannes_disable_bonus',
                            'label' => __('Disable bonus', 'dahannes-connector'),
                            'value' => $disable_bonus,
                            'description' => __('Do not include this product in bonus calculations.', 'dahannes-connector'),
                        )
                    );
                    ?>
                </div>
            </div>
            <?php
            
        }
    }

    public static function dahannes_add_custom_fields(): void
    {
        // This method is still called by the original hook, but we won't add content here anymore
        // It is kept for backward compatibility only
        // All custom fields are now displayed in the custom tab
    }
    
    // This function should be called during plugin initialization
    public static function register_woocommerce_hooks(): void
    {
        // Register the tab
        add_filter('woocommerce_product_data_tabs', [self::class, 'register_product_tabs']);
        
        // Add content to the tab
        add_action('woocommerce_product_data_panels', [self::class, 'add_product_tab_content']);
        
        // Add CSS for the custom tab icon
        add_action('admin_head', function() {
            // Create a mask-image with currentColor to match tab text color
            ?>
            <style type="text/css">
                #woocommerce-product-data ul.wc-tabs li.dahannes-icon a::before {
                    content: '';
                    background-color: currentColor;
                    -webkit-mask-image: url('<?php echo DahannesConnector::get_menu_icon(); ?>');
                    mask-image: url('<?php echo DahannesConnector::get_menu_icon(); ?>');
                    -webkit-mask-repeat: no-repeat;
                    mask-repeat: no-repeat;
                    -webkit-mask-position: center center;
                    mask-position: center center;
                    -webkit-mask-size: 16px 16px;
                    mask-size: 16px 16px;
                    width: 16px;
                    height: 16px;
                    display: inline-block;
                    vertical-align: middle;
                }
            </style>
            <?php
        });
    }
    
    /**
     * Save custom product fields
     *
     * @param int $post_id Product ID
     */
    public static function dahannes_save_custom_fields(int $post_id): void
    {
        // Check if this is a product
        $product = wc_get_product($post_id);
        if (!$product) {
            DahannesLogger::log('error', 'Failed to save DaHannes product fields - product not found', [
                'post_id' => $post_id
            ]);
            return;
        }

        // Get shop type to determine which fields should be saved
        $info_data = get_option('dahannes_info', []);
        $shop_type = $info_data['app_type'] ?? '';
        

        // For commercial shops, save qualification field if it's set
        if ($shop_type === 'commercial' && isset($_POST['_dahannes_gives_qualification'])) {
            $qualification = sanitize_text_field($_POST['_dahannes_gives_qualification']);
            $result = update_post_meta($post_id, '_dahannes_gives_qualification', $qualification);
            
            DahannesLogger::log('info', 'Updated product qualification field', [
                'post_id' => $post_id,
                'qualification' => $qualification,
                'update_result' => $result ? 'success' : 'no change or error'
            ]);
        }
        
        // For both commercial and association shops, save disable_bonus field
        if ($shop_type === 'commercial' || $shop_type === 'association') {
            // Save disable_bonus field (checkbox)
            $disable_bonus = isset($_POST['_dahannes_disable_bonus']) ? 'yes' : 'no';
            $result = update_post_meta($post_id, '_dahannes_disable_bonus', $disable_bonus);
            
            DahannesLogger::log('info', 'Updated product disable bonus field', [
                'post_id' => $post_id,
                'disable_bonus' => $disable_bonus,
                'update_result' => $result ? 'success' : 'no change or error'
            ]);
        }
    }



    public static function register_shortcodes(): void
    {
        // Add the shortcode to WordPress
        add_shortcode('mydahannes_link', function () {
            // The base URL for the link
            $full_url = 'https://my.dahannes.com';


            // Sanitize the referral code to avoid any unwanted characters
            $ref_code = DahannesConnectorUtils::get_member_code();

            if (!empty($ref_code)) {
                // Create the full URL with the referral code
                $full_url .= '?sponsor=' . $ref_code;
            }

            $esc_url = esc_url($full_url);
            return '<a href="' . $esc_url . '" class="uagb-cta_button-link-wrapper wp-block-button_link" target="_blank" rel="noopener noreferrer">Jetzt Mitglied werden<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M504.3 273.6l-112.1 104c-6.992 6.484-17.18 8.218-25.94 4.406c-8.758-3.812-14.42-12.45-14.42-21.1L351.9 288H32C14.33 288 .0002 273.7 .0002 255.1S14.33 224 32 224h319.9l0-72c0-9.547 5.66-18.19 14.42-22c8.754-3.809 18.95-2.075 25.94 4.41l112.1 104C514.6 247.9 514.6 264.1 504.3 273.6z"></path></svg></a>';
        });
    }



    public static function add_dahannes_order_metabox(): void
    {
        $info_data = get_option('dahannes_info', '');
        if (empty($info_data)) {
            return;
        }

        // Don't show meta box for partner shops
        if (isset($info_data['app_type']) && $info_data['app_type'] === 'partner') {
            return;
        }

        // Hook into the WooCommerce admin order page
        add_action('add_meta_boxes', function () {
            add_meta_box(
                'woocommerce_order_dahannes_meta',      // Unique ID for the meta box
                'Dahannes Info zur Bestellung',         // Title of the meta box
                function ($post) {
                    $order = wc_get_order($post->ID);
                    $debug_output = array(
                        'Dahannes Code' => self::get_member_code_from_order($post->ID),
                        'Dahannes Qualifiziert für' => self::get_order_meta_hpos_safe($order, self::DAHANNES_QUAL_SLUG, true) ?: 'Keine',
                    );

                    // Output the information in the meta box
                    echo '<ul>';
                    foreach ($debug_output as $label => $value) {
                        echo '<li><strong>' . esc_html($label) . ':</strong> ' . wp_kses_post($value) . '</li>';
                    }
                    echo '</ul>';
                },
                'shop_order',                        // Admin page to display it on
                'normal',                            // Context
                'low'                                // Priority
            );
        });
    }

}
