<?php

namespace DaHannesConnector\includes;

class DahannesCronManager {
    const CRON_HOOK = 'dahannes_sync_orders';
    const LOCK_TRANSIENT = 'dahannes_sync_lock';
    
    public function __construct() {
        add_filter('cron_schedules', [$this, 'add_cron_interval']);
        add_action(self::CRON_HOOK, [$this, 'sync_callback']);
        
        // Ensure the cron is scheduled when the class is instantiated
        $this->schedule_sync();
    }
    
    public function add_cron_interval($schedules): array {
        $schedules['dahannes_interval'] = [
            'interval' => 300,
            'display' => __('Every 5 minutes', 'dahannes-connector')
        ];
        return $schedules;
    }
    
    public function schedule_sync(): void {
        if (!wp_next_scheduled(self::CRON_HOOK)) {
            wp_schedule_event(time(), 'dahannes_interval', self::CRON_HOOK);
        }
    }
    
    public function sync_callback(): void {
        if ($this->is_sync_locked()) {
            DahannesLogger::log('warning', 'Sync already in progress, skipping');
            return;
        }

        $this->set_sync_lock();

        try {
            DahannesConnectorUtils::update_info_from_dahannes_api();
            DahannesOrderManager::send_orders_to_server();

            DahannesLogger::log('info', 'Cron sync completed successfully');
        } catch (\Exception $e) {
            DahannesLogger::log('error', 'Cron sync failed: ' . $e->getMessage());
        } finally {
            $this->release_sync_lock();
        }
    }
    
    private function is_sync_locked(): bool {
        return (bool)get_transient(self::LOCK_TRANSIENT);
    }
    
    private function set_sync_lock(): void {
        set_transient(self::LOCK_TRANSIENT, true, 300);
    }
    
    private function release_sync_lock(): void {
        delete_transient(self::LOCK_TRANSIENT);
    }
}
