<?php

namespace DaHannesConnector\includes;

use DaHannesConnector;

class DahannesLogger {
    const LOG_OPTION = 'dahannes_recent_logs';
    const MAX_LOGS = 100;
    
    private static $log_file = null;
    private static $sentry_initialized = false;
    private static $installation_context = null;

    /**
     * Initialize Sentry integration if configured
     */
    private static function init_sentry(): void {
        if (self::$sentry_initialized || !self::should_use_sentry()) {
            return;
        }

        $environment = self::get_environment();
        $context = self::get_installation_context();

        try {
            \DaHannesConnectorDeps\Sentry\init([
                'dsn' => self::get_sentry_dsn(),
                'environment' => $environment,
                'release' => 'dahannes-connector@' . DaHannesConnector\PLUGIN_VERSION,
                'traces_sample_rate' => self::get_sentry_sample_rate($environment),
                'send_default_pii' => false,
                'enable_logs' => true, // Enable Sentry logs
                'before_send' => function ($event) use ($context) {
                    // Add installation tags
                    $event->setTag('installation', parse_url($context['site_url'], PHP_URL_HOST));
                    $event->setTag('shop_type', $context['shop_type'] ?? 'unknown');
                    $event->setTag('plugin_version', DaHannesConnector\PLUGIN_VERSION);
                    $event->setTag('wp_version', $context['wp_version']);
                    
                    return $event;
                }
            ]);

            // Configure scope with installation context
            \DaHannesConnectorDeps\Sentry\configureScope(function (\DaHannesConnectorDeps\Sentry\State\Scope $scope) use ($context): void {
                $scope->setContext('installation', $context);
            });

            self::$sentry_initialized = true;
        } catch (\Exception $e) {
            // Silently fail if Sentry setup fails
            error_log('DaHannes: Failed to initialize Sentry: ' . $e->getMessage());
        }
    }

    /**
     * Get Sentry DSN with default fallback
     */
    private static function get_sentry_dsn(): ?string {
        if (defined('DAHANNES_SENTRY_DSN')) {
            return DAHANNES_SENTRY_DSN ?: null;
        }
        
        // Default Sentry DSN
        return 'https://3ee79eea1d69d43e14bce5a13283a122@o4504723831652352.ingest.us.sentry.io/4509962081665024';
    }

    /**
     * Check if Sentry should be used
     */
    private static function should_use_sentry(): bool {
        // Check if Sentry is explicitly disabled
        if (defined('DAHANNES_DISABLE_SENTRY') && DAHANNES_DISABLE_SENTRY) {
            return false;
        }
        
        $dsn = self::get_sentry_dsn();
        return !empty($dsn);
    }

    /**
     * Get Sentry sample rate with defaults based on environment
     */
    private static function get_sentry_sample_rate(string $environment): float {
        // Check for explicit sample rate override
        if (defined('DAHANNES_SENTRY_SAMPLE_RATE')) {
            return (float) DAHANNES_SENTRY_SAMPLE_RATE;
        }
        
        // Default sample rates: lower in production for performance, full in development for debugging
        return $environment === 'production' ? 0.1 : 1.0;
    }

    /**
     * Detect current environment with default fallback
     */
    private static function get_environment(): string {
        // Check for explicit environment override
        if (defined('DAHANNES_ENVIRONMENT') && !empty(DAHANNES_ENVIRONMENT)) {
            return DAHANNES_ENVIRONMENT;
        }
        
        // Auto-detect based on site URL and WordPress debug settings
        $site_url = get_site_url();
        if (strpos($site_url, '.local') !== false || 
            strpos($site_url, '.test') !== false ||
            strpos($site_url, '.dev') !== false ||
            strpos($site_url, 'localhost') !== false) {
            return 'development';
        }
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            return 'development';
        }
        
        // Default to production
        return 'production';
    }

    /**
     * Check if running in development environment
     */
    private static function is_development(): bool {
        return self::get_environment() === 'development';
    }

    /**
     * Get installation context for logging
     */
    private static function get_installation_context(): array {
        if (self::$installation_context !== null) {
            return self::$installation_context;
        }

        $context = [
            'site_url' => get_site_url(),
            'hostname' => gethostname() ?: 'unknown',
            'plugin_version' => DaHannesConnector\PLUGIN_VERSION,
            'wp_version' => get_bloginfo('version'),
            'php_version' => PHP_VERSION,
            'environment' => self::get_environment()
        ];
        
        // Enhance with DaHannes shop info if available
        $shop_info = get_option('dahannes_info', []);
        if (!empty($shop_info) && is_array($shop_info)) {
            $context['shop_name'] = $shop_info['name'] ?? null;
            $context['shop_type'] = $shop_info['app_type'] ?? null;
            $context['shop_url'] = $shop_info['url'] ?? null;
            $context['shop_active'] = $shop_info['is_active'] ?? null;
        }
        
        self::$installation_context = $context;
        return $context;
    }

    /**
     * Check if file logging should be enabled with default fallback
     */
    private static function should_log_to_file(): bool {
        // Check for explicit override
        if (defined('DAHANNES_LOG_TO_FILE')) {
            return (bool) DAHANNES_LOG_TO_FILE;
        }
        
        // Default behavior: enable file logging in development only
        return self::is_development();
    }

    /**
     * Sanitize API key for debugging - shows first and last 4 characters
     */
    private static function sanitize_api_key(string $api_key): string {
        if (strlen($api_key) <= 8) {
            return '[REDACTED]';
        }
        
        return substr($api_key, 0, 4) . '...' . substr($api_key, -4);
    }

    /**
     * Sanitize sensitive data from context while preserving debugging value
     */
    private static function sanitize_context(array $context): array {
        $sensitive_keys = ['password', 'secret', 'dsn'];
        $partial_keys = ['api_key', 'token'];
        
        foreach ($context as $key => $value) {
            if (is_array($value)) {
                $context[$key] = self::sanitize_context($value);
                continue;
            }
            
            if (!is_string($value)) {
                continue;
            }
            
            // Full redaction for highly sensitive keys
            foreach ($sensitive_keys as $sensitive) {
                if (stripos($key, $sensitive) !== false) {
                    $context[$key] = '[REDACTED]';
                    break 2;
                }
            }
            
            // Partial redaction for debugging-useful keys
            foreach ($partial_keys as $partial) {
                if (stripos($key, $partial) !== false) {
                    $context[$key] = self::sanitize_api_key($value);
                    break;
                }
            }
        }
        
        return $context;
    }

    /**
     * Generate a unique request ID for tracking related log entries
     */
    private static function generate_request_id(): string {
        return 'req_' . substr(md5(uniqid(rand(), true)), 0, 8);
    }

    /**
     * Enrich context with installation information and debugging details
     */
    private static function enrich_context(array $context): array {
        $installation_context = self::get_installation_context();
        
        $enriched = [
            'installation_id' => parse_url($installation_context['site_url'], PHP_URL_HOST),
            'timestamp' => current_time('mysql'),
            'memory_usage_mb' => round(memory_get_usage(true) / 1024 / 1024, 2),
        ];
        
        // Add request ID if not already present
        if (!isset($context['request_id'])) {
            $enriched['request_id'] = self::generate_request_id();
        }
        
        return array_merge($enriched, $context);
    }

    /**
     * Enrich API-related context with connection details
     */
    public static function enrich_api_context(array $context, string $server_url = '', string $api_key = ''): array {
        $api_context = [];
        
        if (!empty($server_url)) {
            $api_context['server_url'] = $server_url;
            $api_context['server_host'] = parse_url($server_url, PHP_URL_HOST);
        }
        
        if (!empty($api_key)) {
            $api_context['api_key_preview'] = self::sanitize_api_key($api_key);
        }
        
        // Add environment info relevant to API calls
        $api_context['ssl_verify'] = !self::is_development();
        $api_context['environment'] = self::get_environment();
        
        return array_merge($context, $api_context);
    }

    private static function get_log_file(): string {
        if (self::$log_file === null) {
            self::$log_file = WP_CONTENT_DIR . '/dahannes-connector.log';
        }
        return self::$log_file;
    }

    private static function ensure_log_file(): bool {
        $log_file = self::get_log_file();
        
        // Check if file exists and is writable
        if (file_exists($log_file) && is_writable($log_file)) {
            return true;
        }
        
        // Try to create the file if it doesn't exist
        if (!file_exists($log_file)) {
            $result = @file_put_contents($log_file, '');
            if ($result !== false) {
                @chmod($log_file, 0666);
                return true;
            }
        }
        
        return false;
    }

    /**
     * Main logging method
     */
    public static function log($level, $message, $context = []): void {
        // Initialize Sentry on first log call
        self::init_sentry();
        
        $timestamp = current_time('mysql');
        
        // Sanitize and enrich context
        $context = self::sanitize_context($context);
        $enriched_context = self::enrich_context($context);
        
        $formatted = sprintf('[%s] %s: %s', $timestamp, strtoupper($level), $message);
        if (!empty($context)) {
            $formatted .= ' Context: ' . json_encode($context);
        }
        
        // Log to WordPress error log (errors only)
        if (strtolower($level) === 'error') {
            error_log('DaHannes: ' . $formatted);
        }
        
        // Log to custom file (if enabled)
        if (self::should_log_to_file() && self::ensure_log_file()) {
            @error_log($formatted . PHP_EOL, 3, self::get_log_file());
        }
        
        // Log to Sentry
        if (self::should_use_sentry() && self::$sentry_initialized) {
            self::log_to_sentry($level, $message, $enriched_context);
        }
        
        // Always store recent logs in option for admin UI
        $recent_logs = get_option(self::LOG_OPTION, []);
        array_unshift($recent_logs, [
            'timestamp' => $timestamp,
            'level' => $level,
            'message' => $message,
            'context' => $context
        ]);
        $recent_logs = array_slice($recent_logs, 0, self::MAX_LOGS);
        update_option(self::LOG_OPTION, $recent_logs);
    }

    /**
     * Send log to Sentry using proper Sentry logging API
     */
    private static function log_to_sentry(string $level, string $message, array $context): void {
        try {
            $level_lower = strtolower($level);
            
            // Use the proper logger() function from Sentry
            $logger = \DaHannesConnectorDeps\Sentry\logger();
            
            // Map our log levels to Sentry log levels
            // Note: Sentry expects (message, values_array, attributes_array)
            switch ($level_lower) {
                case 'error':
                    $logger->error($message, [], $context);
                    break;
                case 'warning':
                    $logger->warn($message, [], $context);
                    break;
                case 'info':
                    $logger->info($message, [], $context);
                    break;
                case 'debug':
                    // Only send debug logs in development
                    if (self::is_development()) {
                        $logger->debug($message, [], $context);
                    }
                    break;
                default:
                    $logger->info($message, [], $context);
                    break;
            }
            
            // Force flush logs immediately
            try {
                $eventId = $logger->flush();
                // Only log flush details in development to avoid log spam
                if (self::is_development()) {
                    if ($eventId) {
                        error_log('DaHannes: Sentry logs flushed successfully with event ID: ' . $eventId);
                    } else {
                        error_log('DaHannes: Sentry logs flush returned null (no logs to flush)');
                    }
                }
            } catch (\Exception $flushException) {
                // Always log flush errors since they indicate real issues
                error_log('DaHannes: Failed to flush Sentry logs: ' . $flushException->getMessage());
            }
            
        } catch (\Exception $e) {
            // Silently fail if Sentry logging fails
            error_log('DaHannes: Failed to log to Sentry: ' . $e->getMessage());
        }
    }

    public static function get_recent_logs(): array {
        return get_option(self::LOG_OPTION, []);
    }

    public static function clear_logs(): void {
        delete_option(self::LOG_OPTION);
        if (file_exists(self::get_log_file())) {
            @unlink(self::get_log_file());
        }
    }

    /**
     * Get current environment for external use
     */
    public static function get_current_environment(): string {
        return self::get_environment();
    }

    /**
     * Get installation context for external use
     */
    public static function get_current_installation_context(): array {
        return self::get_installation_context();
    }
}
