<?php

namespace DaHannesConnector\includes;

class DahannesMigrationManager {
    private const MIGRATION_OPTION_KEY = 'dahannes_plugin_version';
    private const BATCH_SIZE = 1000; // Process records in batches to avoid timeouts

    /**
     * Migration registry mapping version to method name
     */
    private static array $migrations = [
        '2.5.3' => 'migrate_cleanup_duplicate_meta',
        '2.6.5' => 'migrate_idempotency_removal_reset',
    ];

    /**
     * Check if HPOS (High Performance Order Storage) is enabled
     */
    private static function isHPOS(): bool {
        return class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') &&
               \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
    }

    /**
     * Get the appropriate meta table name based on HPOS status
     */
    private static function getMetaTableName(): string {
        global $wpdb;
        return self::isHPOS() ? $wpdb->prefix . 'wc_orders_meta' : $wpdb->postmeta;
    }

    /**
     * Get the appropriate order ID column name based on HPOS status
     */
    private static function getOrderIdColumn(): string {
        return self::isHPOS() ? 'order_id' : 'post_id';
    }

    /**
     * Execute migration in batches to avoid timeouts on large datasets
     */
    private static function executeBatchUpdate(string $operation, string $sql, array $params = []): int {
        global $wpdb;

        $total_updated = 0;
        $batch_count = 0;

        do {
            $batch_count++;

            // Add LIMIT to the SQL query
            $batch_sql = rtrim($sql) . " LIMIT " . self::BATCH_SIZE;

            if (!empty($params)) {
                $updated = $wpdb->query($wpdb->prepare($batch_sql, ...$params));
            } else {
                $updated = $wpdb->query($batch_sql);
            }

            if ($updated === false) {
                throw new \Exception("Batch {$batch_count} failed for operation {$operation}");
            }

            $total_updated += (int) $updated;

            DahannesLogger::log('debug', "Batch {$batch_count} completed for {$operation}", [
                'operation' => $operation,
                'batch_number' => $batch_count,
                'batch_size' => $updated,
                'total_processed' => $total_updated
            ]);

            // Continue while we're still processing records
        } while ($updated > 0);

        return $total_updated;
    }
    
    public static function maybe_run_migrations(): void {
        $current_version = \DaHannesConnector\PLUGIN_VERSION;
        $installed_version = get_option(self::MIGRATION_OPTION_KEY, '0.0.0');
        
        if (version_compare($installed_version, $current_version, '<')) {
            self::run_migrations($installed_version, $current_version);
            update_option(self::MIGRATION_OPTION_KEY, $current_version);
        }
    }
    
    private static function run_migrations(string $from_version, string $to_version): void {
        DahannesLogger::log('info', 'Running plugin migrations', [
            'operation' => 'migration',
            'from_version' => $from_version,
            'to_version' => $to_version,
            'available_migrations' => array_keys(self::$migrations)
        ]);

        $executed_migrations = 0;

        foreach (self::$migrations as $migration_version => $migration_method) {
            if (self::shouldRunMigration($from_version, $to_version, $migration_version)) {
                DahannesLogger::log('info', "Executing migration for version {$migration_version}", [
                    'operation' => 'migration',
                    'migration_version' => $migration_version,
                    'migration_method' => $migration_method
                ]);

                try {
                    self::$migration_method();
                    $executed_migrations++;

                    DahannesLogger::log('info', "Migration {$migration_version} completed successfully", [
                        'operation' => 'migration',
                        'migration_version' => $migration_version
                    ]);
                } catch (\Exception $e) {
                    DahannesLogger::log('error', "Migration {$migration_version} failed", [
                        'operation' => 'migration',
                        'migration_version' => $migration_version,
                        'error' => $e->getMessage()
                    ]);
                    throw new \Exception("Migration {$migration_version} failed: " . $e->getMessage());
                }
            }
        }

        DahannesLogger::log('info', 'All migrations completed', [
            'operation' => 'migration',
            'executed_migrations' => $executed_migrations
        ]);
    }

    /**
     * Determine if a migration should run based on version comparison
     */
    private static function shouldRunMigration(string $from_version, string $to_version, string $migration_version): bool {
        return version_compare($from_version, $migration_version, '<') &&
               version_compare($to_version, $migration_version, '>=');
    }
    
    private static function migrate_cleanup_duplicate_meta(): void {
        global $wpdb;

        DahannesLogger::log('info', 'Starting duplicate meta cleanup migration', [
            'operation' => 'migration_duplicate_cleanup',
            'target_version' => '2.5.3'
        ]);

        $start_time = microtime(true);
        $meta_table = self::getMetaTableName();

        try {
            // For safety, we'll use separate queries for each table type to avoid injection
            if (self::isHPOS()) {
                // HPOS table - use 'order_id' column and 'id' as primary key
                $deleted = $wpdb->query($wpdb->prepare(
                    "DELETE m1 FROM {$wpdb->prefix}wc_orders_meta m1
                     INNER JOIN {$wpdb->prefix}wc_orders_meta m2
                     WHERE m1.id < m2.id
                     AND m1.order_id = m2.order_id
                     AND m1.meta_key = m2.meta_key
                     AND m1.meta_key IN (%s, %s)",
                    DahannesConnectorUtils::DAHANNES_SUBMISSION_STATUS,
                    DahannesConnectorUtils::DAHANNES_PAYLOAD_HASH
                ));
            } else {
                // Legacy postmeta table - use 'post_id' column and 'meta_id' as primary key
                $deleted = $wpdb->query($wpdb->prepare(
                    "DELETE m1 FROM {$wpdb->postmeta} m1
                     INNER JOIN {$wpdb->postmeta} m2
                     WHERE m1.meta_id < m2.meta_id
                     AND m1.post_id = m2.post_id
                     AND m1.meta_key = m2.meta_key
                     AND m1.meta_key IN (%s, %s)",
                    DahannesConnectorUtils::DAHANNES_SUBMISSION_STATUS,
                    DahannesConnectorUtils::DAHANNES_PAYLOAD_HASH
                ));
            }

            if ($deleted === false) {
                throw new \Exception('Database query failed');
            }

            $duration = round(microtime(true) - $start_time, 2);

            DahannesLogger::log('info', 'Duplicate meta cleanup migration completed', [
                'operation' => 'migration_duplicate_cleanup',
                'records_deleted' => $deleted,
                'duration_seconds' => $duration,
                'table_used' => basename($meta_table),
                'is_hpos' => self::isHPOS()
            ]);
        } catch (\Exception $e) {
            $duration = round(microtime(true) - $start_time, 2);

            DahannesLogger::log('error', 'Failed to cleanup duplicate meta', [
                'operation' => 'migration_duplicate_cleanup',
                'error' => $e->getMessage(),
                'duration_seconds' => $duration,
                'table_used' => basename($meta_table),
                'is_hpos' => self::isHPOS()
            ]);
            throw $e;
        }
    }
    
    public static function reset_migration_version(): void {
        delete_option(self::MIGRATION_OPTION_KEY);
        DahannesLogger::log('info', 'Migration version reset (for testing)', [
            'operation' => 'reset_migration_version'
        ]);
    }
    
    public static function get_migration_version(): string {
        return get_option(self::MIGRATION_OPTION_KEY, '0.0.0');
    }

    /**
     * Migration for 2.6.5: Reset submission status after idempotency mechanism removal
     *
     * This migration resets all order sync status to ensure orders are resubmitted
     * with updated data after removing the buggy idempotency mechanism. The idempotency
     * key was causing data consistency issues where modified orders within the same hour
     * would not be resubmitted with their updated state.
     */
    private static function migrate_idempotency_removal_reset(): void {
        DahannesLogger::log('info', 'Starting idempotency removal migration', [
            'operation' => 'migration_idempotency_removal',
            'target_version' => '2.6.5',
            'description' => 'Reset submission status after removing idempotency mechanism',
            'reason' => 'Idempotency key caused data consistency bugs for modified orders'
        ]);

        $start_time = microtime(true);
        $meta_table = self::getMetaTableName();
        $table_type = self::isHPOS() ? 'HPOS' : 'Legacy';

        try {
            // Reset all submission status to trigger reprocessing without idempotency caching
            $sql = "UPDATE {$meta_table}
                    SET meta_value = '0'
                    WHERE meta_key = %s
                    AND meta_value != '0'";

            $updated = self::executeBatchUpdate(
                'idempotency_removal_reset',
                $sql,
                [DahannesConnectorUtils::DAHANNES_SUBMISSION_STATUS]
            );

            DahannesLogger::log('info', "Reset submission status after idempotency removal ({$table_type})", [
                'operation' => 'migration_idempotency_removal',
                'updated_orders' => $updated,
                'table' => basename($meta_table),
                'duration_ms' => round((microtime(true) - $start_time) * 1000, 2),
                'reason' => 'Ensure all orders resubmit with current data, no idempotency caching'
            ]);
        } catch (\Exception $e) {
            DahannesLogger::log('error', 'Failed to reset submission status for idempotency removal', [
                'operation' => 'migration_idempotency_removal',
                'error' => $e->getMessage(),
                'table' => basename($meta_table),
                'duration_ms' => round((microtime(true) - $start_time) * 1000, 2)
            ]);
            throw $e;
        }

        // Clear payload hashes to ensure all orders are properly recalculated
        self::clear_payload_hashes();

        DahannesLogger::log('info', 'Idempotency removal migration completed', [
            'operation' => 'migration_idempotency_removal',
            'total_updated' => $updated,
            'total_duration_ms' => round((microtime(true) - $start_time) * 1000, 2),
            'benefit' => 'All orders will be resubmitted with current data without idempotency bugs'
        ]);
    }

    /**
     * Clear payload hashes to force recalculation
     */
    private static function clear_payload_hashes(): void {
        $meta_table = self::getMetaTableName();

        try {
            $sql = "DELETE FROM {$meta_table}
                    WHERE meta_key = %s";

            $result = self::executeBatchUpdate(
                'clear_payload_hashes',
                $sql,
                [DahannesConnectorUtils::DAHANNES_PAYLOAD_HASH]
            );

            DahannesLogger::log('debug', 'Cleared payload hashes during migration', [
                'operation' => 'migration_idempotency_removal',
                'records_deleted' => $result,
                'reason' => 'Force recalculation without idempotency mechanism'
            ]);
        } catch (\Exception $e) {
            DahannesLogger::log('error', 'Failed to clear payload hashes', [
                'operation' => 'migration_idempotency_removal',
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }
}