<?php

namespace DaHannesConnector\includes;

class DahannesOrderManager {
    private ApiClient $api_client;
    private int $batch_size;
    
    public function __construct(ApiClient $api_client = null, int $batch_size = 50) {
        $this->api_client = $api_client ?? new ApiClient(\DaHannesConnector\PLUGIN_VERSION);
        $this->batch_size = $batch_size;
    }

    /**
     * Log order IDs in batches of 5 to reduce log volume
     */
    private function log_order_ids_in_batches(array $order_ids, string $message, string $operation, int $batch_number): void {
        $chunks = array_chunk($order_ids, 5);
        foreach ($chunks as $chunk) {
            DahannesLogger::log('info', $message . ': ' . implode(', ', $chunk), [
                'operation' => $operation,
                'batch_number' => $batch_number,
                'order_ids' => $chunk
            ]);
        }
    }

    public function process_orders(): array {
        if (!$this->api_client->api_credentials_available()) {
            DahannesLogger::log('error', 'API credentials not available for order processing', [
                'operation' => 'process_orders',
                'batch_size' => $this->batch_size,
                'server_url' => get_option('dahannes_server_url'),
                'api_key_configured' => !empty(get_option('api_key'))
            ]);
            return ['success' => 0, 'failed' => 0];
        }

        // Check if account is active and data_version is available
        $info_data = get_option('dahannes_info', []);
        if (isset($info_data['is_active']) && !$info_data['is_active']) {
            DahannesLogger::log('warning', 'Account is inactive. Bonuses will not be submitted.', [
                'operation' => 'process_orders',
                'account_status' => 'inactive',
                'shop_info' => array_intersect_key($info_data, array_flip(['name', 'app_type', 'data_version'])),
                'batch_size' => $this->batch_size
            ]);
            return ['success' => 0, 'failed' => 0];
        }

        // Validate data_version is available for order marking
        $data_version = $info_data['data_version'] ?? '';
        if (empty($data_version)) {
            DahannesLogger::log('error', 'Cannot process orders: missing data_version from DaHannes API', [
                'operation' => 'process_orders',
                'info_data_keys' => array_keys($info_data),
                'has_data_version' => isset($info_data['data_version']),
                'data_version_value' => $info_data['data_version'] ?? null,
                'batch_size' => $this->batch_size
            ]);
            return ['success' => 0, 'failed' => 0];
        }

        // Get ALL pending orders without limit
        $orders = $this->get_pending_orders();

        if (empty($orders)) {
            DahannesLogger::log('debug', 'No pending orders found for processing', [
                'operation' => 'process_orders',
                'batch_size' => $this->batch_size
            ]);
            return ['success' => 0, 'failed' => 0];
        }

        // Sort orders by ID (oldest first)
        ksort($orders);

        DahannesLogger::log('info', 'Starting order processing', [
            'operation' => 'process_orders',
            'total_orders' => count($orders),
            'batch_size' => $this->batch_size,
            'estimated_batches' => ceil(count($orders) / $this->batch_size)
        ]);

        // Process orders in controlled batches
        $success = 0;
        $failed = 0;
        $batch = [];
        $count = 0;

        foreach ($orders as $order_id => $order_data) {
            $batch[$order_id] = $order_data;
            $count++;

            // When we reach batch_size, process the batch
            if ($count >= $this->batch_size) {
                $batch_number = ceil(($success + $failed + count($batch)) / $this->batch_size);

                DahannesLogger::log('debug', 'Sending batch of orders', [
                    'operation' => 'process_orders',
                    'batch_number' => $batch_number,
                    'batch_size' => count($batch),
                    'order_ids' => array_keys($batch),
                    'cumulative_processed' => $success + $failed
                ]);

                try {
                    $result = $this->api_client->send_order_list($batch);

                    if ($result) {
                        $mark_result = $this->mark_orders_as_processed(array_keys($batch));

                        if (!empty($mark_result['failed'])) {
                            // If marking failed, don't count as success
                            $failed += count($mark_result['failed']);
                            $success += $mark_result['success'];

                            DahannesLogger::log('error', 'Orders sent to API but failed to mark as processed', [
                                'operation' => 'process_orders',
                                'batch_number' => $batch_number,
                                'api_success' => true,
                                'marking_failed_count' => count($mark_result['failed']),
                                'marking_success_count' => $mark_result['success'],
                                'failed_ids' => $mark_result['failed'],
                                'sample_errors' => array_slice($mark_result['errors'], 0, 3)
                            ]);
                        } else {
                            $success += count($batch);
                        }
                        
                        DahannesLogger::log('info', 'Successfully processed order batch', [
                            'operation' => 'process_orders',
                            'batch_number' => $batch_number,
                            'batch_size' => count($batch),
                            'order_ids' => array_keys($batch),
                            'cumulative_success' => $success,
                            'cumulative_failed' => $failed
                        ]);

                        // Log successfully synced order IDs in batches of 5
                        $this->log_order_ids_in_batches(
                            array_keys($batch),
                            'Successfully synced order IDs',
                            'orders_synced',
                            $batch_number
                        );
                    } else {
                        $failed += count($batch);
                        DahannesLogger::log('error', 'Failed to process order batch', [
                            'operation' => 'process_orders',
                            'batch_number' => $batch_number,
                            'batch_size' => count($batch),
                            'order_ids' => array_keys($batch),
                            'cumulative_success' => $success,
                            'cumulative_failed' => $failed,
                            'api_client_result' => $result
                        ]);
                    }
                } catch (\Exception $e) {
                    $failed += count($batch);
                    DahannesLogger::log('error', 'Exception processing order batch', [
                        'operation' => 'process_orders',
                        'batch_number' => $batch_number,
                        'batch_size' => count($batch),
                        'order_ids' => array_keys($batch),
                        'exception_message' => $e->getMessage(),
                        'exception_code' => $e->getCode(),
                        'exception_file' => $e->getFile(),
                        'exception_line' => $e->getLine(),
                        'cumulative_success' => $success,
                        'cumulative_failed' => $failed
                    ]);
                }
                
                // Reset for next batch
                $batch = [];
                $count = 0;
            }
        }
        
        // Process any remaining orders
        if (!empty($batch)) {
            $batch_number = ceil(($success + $failed + count($batch)) / $this->batch_size);
            
            DahannesLogger::log('debug', 'Sending final batch of orders', [
                'operation' => 'process_orders',
                'batch_number' => $batch_number,
                'batch_size' => count($batch),
                'order_ids' => array_keys($batch),
                'cumulative_processed' => $success + $failed,
                'final_batch' => true
            ]);

            try {
                $result = $this->api_client->send_order_list($batch);

                if ($result) {
                    $mark_result = $this->mark_orders_as_processed(array_keys($batch));

                    if (!empty($mark_result['failed'])) {
                        // If marking failed, don't count as success
                        $failed += count($mark_result['failed']);
                        $success += $mark_result['success'];

                        DahannesLogger::log('error', 'Final batch sent to API but failed to mark as processed', [
                            'operation' => 'process_orders',
                            'batch_number' => $batch_number,
                            'api_success' => true,
                            'marking_failed_count' => count($mark_result['failed']),
                            'marking_success_count' => $mark_result['success'],
                            'failed_ids' => $mark_result['failed'],
                            'sample_errors' => array_slice($mark_result['errors'], 0, 3),
                            'final_batch' => true
                        ]);
                    } else {
                        $success += count($batch);
                    }
                    
                    DahannesLogger::log('info', 'Successfully processed final order batch', [
                        'operation' => 'process_orders',
                        'batch_number' => $batch_number,
                        'batch_size' => count($batch),
                        'order_ids' => array_keys($batch),
                        'cumulative_success' => $success,
                        'cumulative_failed' => $failed,
                        'final_batch' => true
                    ]);

                    // Log successfully synced order IDs in batches of 5
                    $this->log_order_ids_in_batches(
                        array_keys($batch),
                        'Successfully synced order IDs',
                        'orders_synced',
                        $batch_number
                    );
                } else {
                    $failed += count($batch);
                    DahannesLogger::log('error', 'Failed to process final order batch', [
                        'operation' => 'process_orders',
                        'batch_number' => $batch_number,
                        'batch_size' => count($batch),
                        'order_ids' => array_keys($batch),
                        'cumulative_success' => $success,
                        'cumulative_failed' => $failed,
                        'api_client_result' => $result,
                        'final_batch' => true
                    ]);
                }
            } catch (\Exception $e) {
                $failed += count($batch);
                DahannesLogger::log('error', 'Exception processing final order batch', [
                    'operation' => 'process_orders',
                    'batch_number' => $batch_number,
                    'batch_size' => count($batch),
                    'order_ids' => array_keys($batch),
                    'exception_message' => $e->getMessage(),
                    'exception_code' => $e->getCode(),
                    'exception_file' => $e->getFile(),
                    'exception_line' => $e->getLine(),
                    'cumulative_success' => $success,
                    'cumulative_failed' => $failed,
                    'final_batch' => true
                ]);
            }
        }
        
        // Log processing summary
        $total_orders = $success + $failed;
        $log_level = $failed > 0 ? 'warning' : 'info';

        DahannesLogger::log($log_level, 'Order processing completed', [
            'operation' => 'process_orders',
            'total_orders_processed' => $total_orders,
            'successful_orders' => $success,
            'failed_orders' => $failed,
            'success_rate' => $total_orders > 0 ? round(($success / $total_orders) * 100, 2) : 0,
            'batch_size_used' => $this->batch_size
        ]);

        return ['success' => $success, 'failed' => $failed];
    }

    private function get_pending_orders(): array {
        // Get ALL pending orders (global sync lock prevents concurrent access)
        $all_orders = DahannesConnectorUtils::compile_list_of_orders_for_server();
        
        // Limit to match compile_list_of_orders_for_server limit
        $max_orders = 100;
        
        if (count($all_orders) > $max_orders) {
            DahannesLogger::log('info', 'Limiting order processing to prevent long-running process', [
                'total_pending' => count($all_orders),
                'max_per_run' => $max_orders,
                'operation' => 'get_pending_orders'
            ]);
            
            // Take only the first N orders (oldest first, as they're sorted by ID)
            $all_orders = array_slice($all_orders, 0, $max_orders, true);
        }
        
        return $all_orders;
    }

    private function mark_orders_as_processed(array $order_ids): array {
        $info_data = get_option('dahannes_info', []);
        $data_version = $info_data['data_version'] ?? '';

        if (empty($data_version)) {
            DahannesLogger::log('error', 'Cannot mark orders as processed: missing data_version', [
                'operation' => 'mark_orders_as_processed',
                'order_ids' => $order_ids,
                'info_data_keys' => array_keys($info_data),
                'has_data_version' => isset($info_data['data_version']),
                'data_version_value' => $info_data['data_version'] ?? null
            ]);
            return ['success' => 0, 'failed' => $order_ids, 'errors' => array_fill_keys($order_ids, 'Missing data_version')];
        }

        return DahannesConnectorUtils::mark_orders_processed($order_ids, $data_version);
    }

    public function resync_orders(array $order_ids): array {
        foreach ($order_ids as $order_id) {
            $order = wc_get_order($order_id);
            if ($order) {
                $order->update_meta_data(DahannesConnectorUtils::DAHANNES_SUBMISSION_STATUS, '0');
                $order->save();
            }
        }
        return $this->process_orders();
    }

    /**
     * Static entry point for cron jobs
     * Replaces DahannesConnectorUtils::send_orders_to_server()
     */
    public static function send_orders_to_server(): void {
        $manager = new self();
        $result = $manager->process_orders();
        
        DahannesLogger::log('info', 'Order processing completed via cron', [
            'success' => $result['success'],
            'failed' => $result['failed']
        ]);
    }

}
