<?php
/**
 * File: plugin/includes/wp-cli-commands.php
 * Description: WP-CLI commands for DaHannes Connector
 */

namespace DaHannesConnector\includes;

use WC_Order;
use WP_CLI;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (!class_exists('WP_CLI')) {
    return;
}

/**
 * Manages DaHannes Connector order synchronization.
 */
class DahannesCLICommands {

    /**
     * Resets the synchronization status of orders to trigger resubmission.
     *
     * ## OPTIONS
     *
     * [--order_id=<order_id>]
     * : Reset specific order ID. If not provided, resets all orders.
     *
     * [--status=<status>]
     * : Filter by order status (completed, processing, etc). Can be comma-separated for multiple statuses.
     *
     * [--limit=<limit>]
     * : Maximum number of orders to reset. Default: no limit.
     *
     * [--dry-run]
     * : Show how many orders would be affected without making changes.
     *
     * ## EXAMPLES
     *
     *     # Reset all orders
     *     $ wp dahannes reset_sync
     *
     *     # Reset a specific order
     *     $ wp dahannes reset_sync --order_id=123
     *
     *     # Reset only completed orders
     *     $ wp dahannes reset_sync --status=completed
     *
     *     # Reset completed and processing orders, limit to 50
     *     $ wp dahannes reset_sync --status=completed,processing --limit=50
     *
     * @param array $args Command arguments
     * @param array $assoc_args Command options
     */
    public function reset_sync($args, $assoc_args) {
        // Parse arguments
        $order_id = isset($assoc_args['order_id']) ? intval($assoc_args['order_id']) : null;
        $status = isset($assoc_args['status']) ? explode(',', $assoc_args['status']) : null;
        $limit = isset($assoc_args['limit']) ? intval($assoc_args['limit']) : -1;
        $dry_run = isset($assoc_args['dry-run']);

        // Build query args
        $query_args = [
            'limit' => $limit,
            'return' => 'ids',
        ];

        // Add order ID filter if specified
        if ($order_id) {
            $query_args['include'] = [$order_id];
        }

        // Add status filter if specified
        if ($status) {
            $query_args['status'] = $status;
        }

        // Get orders
        $query = new \WC_Order_Query($query_args);
        $order_ids = $query->get_orders();

        if (empty($order_ids)) {
            WP_CLI::warning('No orders found matching the criteria.');
            return;
        }

        WP_CLI::log(sprintf('Found %d orders to process.', count($order_ids)));

        if ($dry_run) {
            WP_CLI::success('Dry run completed. No changes made.');
            return;
        }

        // Create progress bar
        $progress = \WP_CLI\Utils\make_progress_bar('Resetting order sync status', count($order_ids));
        $count = 0;

        // Fast atomic reset using direct database query
        global $wpdb;
        
        // HPOS compatible - check if orders are stored in custom tables
        $is_hpos = wc_get_container()->get(\Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController::class)->custom_orders_table_usage_is_enabled();
        
        if ($is_hpos) {
            // HPOS: Update order meta in the custom orders table
            $meta_table = $wpdb->prefix . 'wc_orders_meta';
            $placeholders = implode(',', array_fill(0, count($order_ids), '%d'));
            
            $updated = $wpdb->query(
                $wpdb->prepare(
                    "UPDATE {$meta_table} 
                     SET meta_value = '0' 
                     WHERE meta_key = %s 
                     AND order_id IN ({$placeholders})",
                    DahannesConnectorUtils::DAHANNES_SUBMISSION_STATUS,
                    ...$order_ids
                )
            );
        } else {
            // Legacy: Update order meta in postmeta table  
            $placeholders = implode(',', array_fill(0, count($order_ids), '%d'));
            
            $updated = $wpdb->query(
                $wpdb->prepare(
                    "UPDATE {$wpdb->postmeta} 
                     SET meta_value = '0' 
                     WHERE meta_key = %s 
                     AND post_id IN ({$placeholders})",
                    DahannesConnectorUtils::DAHANNES_SUBMISSION_STATUS,
                    ...$order_ids
                )
            );
        }
        
        $progress->finish();
        
        if ($updated !== false) {
            WP_CLI::success(sprintf('Reset sync status for %d orders in single atomic operation.', count($order_ids)));
        } else {
            WP_CLI::error('Database update failed.');
        }
    }

    /**
     * Repairs missing dahannes_qualifies_for meta on existing orders and marks them for resubmission.
     *
     * ## OPTIONS
     *
     * [--order_id=<order_id>]
     * : Repair specific order ID. If not provided, checks all orders.
     *
     * [--status=<status>]
     * : Filter by order status (completed, processing, etc). Can be comma-separated for multiple statuses.
     *
     * [--limit=<limit>]
     * : Maximum number of orders to check. Default: no limit.
     *
     * [--start-date=<start-date>]
     * : Only check orders created after this date (YYYY-MM-DD format). Default: 2024-10-01.
     *
     * [--dry-run]
     * : Show how many orders would be repaired without making changes.
     *
     * ## EXAMPLES
     *
     *     # Check and repair all orders (dry run first)
     *     $ wp dahannes repair_qualifications --dry-run
     *     $ wp dahannes repair_qualifications
     *
     *     # Repair a specific order
     *     $ wp dahannes repair_qualifications --order_id=123
     *
     *     # Repair only completed orders
     *     $ wp dahannes repair_qualifications --status=completed
     *
     *     # Repair orders from a specific date
     *     $ wp dahannes repair_qualifications --start-date=2024-12-01
     *
     * @param array $args Command arguments
     * @param array $assoc_args Command options
     */
    public function repair_qualifications($args, $assoc_args) {
        // Parse arguments
        $order_id = isset($assoc_args['order_id']) ? intval($assoc_args['order_id']) : null;
        $status = isset($assoc_args['status']) ? explode(',', $assoc_args['status']) : ['completed', 'processing', 'on-hold', 'pending'];
        $limit = isset($assoc_args['limit']) ? intval($assoc_args['limit']) : -1;
        $start_date = isset($assoc_args['start-date']) ? $assoc_args['start-date'] : '2024-10-01';
        $dry_run = isset($assoc_args['dry-run']);

        // Build query args - only get orders with member codes
        $query_args = [
            'limit' => $limit,
            'status' => $status,
            'meta_query' => [
                [
                    'key' => 'dahannes_member_code',
                    'compare' => 'EXISTS'
                ]
            ]
        ];

        // Add order ID filter if specified
        if ($order_id) {
            $query_args['include'] = [$order_id];
            unset($query_args['status']); // Remove status filter when targeting specific order
        } else {
            // Add date filter if not targeting specific order
            try {
                $date_timestamp = strtotime($start_date);
                if ($date_timestamp === false) {
                    WP_CLI::error('Invalid start date format. Please use YYYY-MM-DD format.');
                }
                $query_args['date_created'] = '>=' . $date_timestamp;
                WP_CLI::log(sprintf('Checking orders created after: %s', $start_date));
            } catch (Exception $e) {
                WP_CLI::error('Error parsing start date: ' . $e->getMessage());
            }
        }

        // Get orders
        $query = new \WC_Order_Query($query_args);
        $orders = $query->get_orders();

        if (empty($orders)) {
            WP_CLI::warning('No orders found matching the criteria.');
            return;
        }

        WP_CLI::log(sprintf('Found %d orders with member codes to check.', count($orders)));

        $needs_repair = [];
        $already_correct = 0;
        $legacy_correct = 0;

        // First pass: identify orders that need repair
        foreach ($orders as $order) {
            $order_id = $order->get_id();
            
            // Check if order already has qualification meta - HPOS compatible
            $existing_qual = $order->get_meta('dahannes_qualifies_for', true);
            
            // Check if order has any products with qualifications
            $should_have_qual = false;
            $expected_qualifications = [];
            
            foreach ($order->get_items() as $item) {
                $product_id = $item->get_product_id();
                $qualification = get_post_meta($product_id, '_dahannes_gives_qualification', true);
                if (!empty($qualification)) {
                    $should_have_qual = true;
                    $expected_qualifications[] = $qualification;
                }
            }
            
            if ($should_have_qual && empty($existing_qual)) {
                $needs_repair[] = [
                    'order_id' => $order_id,
                    'expected' => implode(',', array_unique($expected_qualifications)),
                    'order_date' => $order->get_date_created()->format('Y-m-d H:i:s')
                ];
            } elseif ($should_have_qual && !empty($existing_qual)) {
                // Check if it has legacy "best-of-all" qualification
                if ($existing_qual === 'best-of-all') {
                    $legacy_correct++;
                } else {
                    $already_correct++;
                }
            }
        }

        WP_CLI::log(sprintf('Orders needing repair: %d', count($needs_repair)));
        WP_CLI::log(sprintf('Orders already correct: %d', $already_correct));
        if ($legacy_correct > 0) {
            WP_CLI::log(sprintf('Orders with legacy "best-of-all" qualification: %d', $legacy_correct));
        }

        if (empty($needs_repair)) {
            WP_CLI::success('No orders need repair.');
            return;
        }

        if ($dry_run) {
            WP_CLI::log("\nOrders that would be repaired:");
            foreach ($needs_repair as $repair) {
                WP_CLI::log(sprintf('- Order #%d (%s) would get qualification: %s', 
                    $repair['order_id'], 
                    $repair['order_date'], 
                    $repair['expected']
                ));
            }
            WP_CLI::success('Dry run completed. No changes made.');
            return;
        }

        // Second pass: actually repair the orders
        $repaired = 0;
        $failed = 0;
        $repaired_orders = [];
        
        if (!empty($needs_repair)) {
            $progress = \WP_CLI\Utils\make_progress_bar('Repairing order qualifications', count($needs_repair));

            foreach ($needs_repair as $repair) {
                $order_id = $repair['order_id'];
                
                // Run the qualification check
                DahannesConnectorUtils::check_member_code_and_product_meta($order_id);
                
                // Verify it worked - HPOS compatible
                $order = wc_get_order($order_id);
                $new_qual = $order ? $order->get_meta('dahannes_qualifies_for', true) : '';
                if (!empty($new_qual)) {
                    // Reset submission status to trigger resubmission - HPOS compatible
                    $order->update_meta_data(DahannesConnectorUtils::DAHANNES_SUBMISSION_STATUS, '0');
                    $order->save();
                    $repaired_orders[] = [
                        'ID' => $order_id,
                        'Date' => $repair['order_date'],
                        'Status' => $order->get_status(),
                        'Qualification' => $new_qual
                    ];
                    $repaired++;
                } else {
                    $failed++;
                }
                
                $progress->tick();
            }

            $progress->finish();
        }
        
        
        // Display table of repaired orders
        if (!empty($repaired_orders)) {
            WP_CLI::log("\nRepaired orders:");
            \WP_CLI\Utils\format_items('table', $repaired_orders, ['ID', 'Date', 'Status', 'Qualification']);
        }
        
        
        WP_CLI::success(sprintf('Repair completed. Success: %d, Failed: %d', $repaired, $failed));
        
        if ($failed > 0) {
            WP_CLI::warning(sprintf('%d orders failed to repair. Check logs for details.', $failed));
        }
    }

    /**
     * Updates legacy "best-of-all" qualifications to "welcome-package".
     *
     * ## OPTIONS
     *
     * [--dry-run]
     * : Show how many orders would be updated without making changes.
     *
     * ## EXAMPLES
     *
     *     # Check how many orders have legacy qualifications
     *     $ wp dahannes fix_legacy_qualifications --dry-run
     *
     *     # Update all legacy qualifications
     *     $ wp dahannes fix_legacy_qualifications
     *
     * @param array $args Command arguments
     * @param array $assoc_args Command options
     */
    public function fix_legacy_qualifications($args, $assoc_args) {
        $dry_run = isset($assoc_args['dry-run']);
        
        global $wpdb;
        
        // Count orders with best-of-all qualification
        $count = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM {$wpdb->postmeta} 
            WHERE meta_key = 'dahannes_qualifies_for' 
            AND meta_value = 'best-of-all'
        ");
        
        if ($count == 0) {
            WP_CLI::success('No orders with legacy "best-of-all" qualification found.');
            return;
        }
        
        WP_CLI::log(sprintf('Found %d orders with legacy "best-of-all" qualification.', $count));
        
        if ($dry_run) {
            WP_CLI::success('Dry run completed. No changes made.');
            return;
        }
        
        // Update all best-of-all to welcome-package
        $updated = $wpdb->update(
            $wpdb->postmeta,
            ['meta_value' => 'welcome-package'],
            [
                'meta_key' => 'dahannes_qualifies_for',
                'meta_value' => 'best-of-all'
            ],
            ['%s'],
            ['%s', '%s']
        );
        
        if ($updated === false) {
            WP_CLI::error('Failed to update legacy qualifications.');
        } else {
            WP_CLI::success(sprintf('Successfully updated %d orders from "best-of-all" to "welcome-package".', $updated));
        }
    }

    /**
     * Exports order IDs and subtotals (minus discounts/refunds) in CSV format.
     *
     * ## OPTIONS
     *
     * [--status=<status>]
     * : Filter by order status (completed, processing, etc). Can be comma-separated for multiple statuses.
     *
     * [--limit=<limit>]
     * : Maximum number of orders to export. Default: no limit.
     *
     * [--days=<days>]
     * : Only include orders from the last X days. Default: all orders.*
     *
     * [--output=<output>]
     * : Print the output in the specified format. Default: csv.
     *
     * ## EXAMPLES
     *
     *     # Export all orders
     *     $ wp dahannes export_orders
     *
     *     # Export only completed orders
     *     $ wp dahannes export_orders --status=completed
     *
     *     # Export completed and processing orders from the last 30 days
     *     $ wp dahannes export_orders --status=completed,processing --days=30
     *
     * @param array $args Command arguments
     * @param array $assoc_args Command options
     */
    public function export_orders($args, $assoc_args)
    {
        // Parse arguments
        $status = isset($assoc_args['status']) ? explode(',', $assoc_args['status']) : null;
        $limit = isset($assoc_args['limit']) ? intval($assoc_args['limit']) : -1;
        $output = isset($assoc_args['output']) ? trim($assoc_args['output']) : 'csv';
        $days = isset($assoc_args['days']) ? intval($assoc_args['days']) : null;

        // Build query args
        $query_args = [
            'limit' => $limit,
            'return' => 'objects',
        ];

        // Add status filter if specified
        if ($status) {
            $query_args['status'] = $status;
        }

        // Add date filter if days specified
        if ($days) {
            $query_args['date_created'] = '>' . (time() - ($days * DAY_IN_SECONDS));
        }

        // Get orders
        $query = new \WC_Order_Query($query_args);
        $orders = $query->get_orders();

        if (empty($orders)) {
            WP_CLI::warning('No orders found matching the criteria.');
            return;
        }

        WP_CLI::log(sprintf('Found %d orders to export.', count($orders)));

        if ($output === 'csv') {
        // Output CSV header
        WP_CLI::line('subtotal,order_id');
    }

        // Process each order
        foreach ($orders as $order) {
            // Skip refund orders
            if ($order->get_type() === 'shop_order_refund') {
                continue;
            }
            
            $order_id = $order->get_id();

            /** @var WC_Order $order */
            $percentage_refunded = $order->get_total_refunded() / $order->get_total();

            $subtotal =$order->get_subtotal();
            $subtotal_not_refunded = $subtotal * (1 - $percentage_refunded);

            if($output === 'csv') {
                // Format to 2 decimal places and output as CSV line
                WP_CLI::line(sprintf('%.2f,%d', $subtotal_not_refunded, $order_id));
            } elseif($output === 'sql') {
                // Output as SQL INSERT statement
                $total_cents = $subtotal_not_refunded * 100;
                WP_CLI::line(sprintf("update shop_orders set order_total = $total_cents where client_app_id = 2 and remote_id = $order_id;"));
            }
        }

        WP_CLI::success(sprintf('Exported %d orders.', count($orders)));
    }

    /**
     * Synchronizes orders to DaHannes server without the 200 order limit.
     *
     * This command processes all pending orders for synchronization, unlike the
     * automatic cron sync which is limited to 200 orders per run for performance.
     *
     * ## OPTIONS
     *
     * [--dry-run]
     * : Show what would be synchronized without actually sending orders.
     *
     * [--batch-size=<size>]
     * : Number of orders to process per batch. Default: 50.
     *
     * [--ignore-delay]
     * : Ignore the 30-second delay period for newly created orders.
     *
     * ## EXAMPLES
     *
     *     # Sync all pending orders (dry run first)
     *     $ wp dahannes sync_orders_unlimited --dry-run
     *     $ wp dahannes sync_orders_unlimited
     *
     *     # Sync with custom batch size
     *     $ wp dahannes sync_orders_unlimited --batch-size=25
     *
     *     # Force sync ignoring delay period
     *     $ wp dahannes sync_orders_unlimited --ignore-delay
     *
     * @param array $args Command arguments
     * @param array $assoc_args Command options
     */
    public function sync_orders_unlimited($args, $assoc_args) {
        $dry_run = isset($assoc_args['dry-run']);
        $batch_size = isset($assoc_args['batch-size']) ? intval($assoc_args['batch-size']) : 50;
        $ignore_delay = isset($assoc_args['ignore-delay']);

        // Validate batch size
        if ($batch_size < 1 || $batch_size > 200) {
            WP_CLI::error('Batch size must be between 1 and 200.');
            return;
        }

        WP_CLI::log('Starting unlimited order synchronization...');

        // Check API credentials
        $client = new ApiClient();
        if (!$client->api_credentials_available()) {
            WP_CLI::error('API credentials not available. Please configure the plugin first.');
            return;
        }
        
        // Update shop info from API first (same as cron job)
        try {
            DahannesConnectorUtils::update_info_from_dahannes_api();
        } catch (\Exception $e) {
            WP_CLI::warning('Failed to update shop info from API: ' . $e->getMessage());
        }

        // Check if account is active
        $info_data = get_option('dahannes_info', []);
        if (isset($info_data['is_active']) && !$info_data['is_active']) {
            WP_CLI::error('Account is inactive. Bonuses cannot be submitted.');
            return;
        }

        // Get all orders that need synchronization
        $orders = DahannesConnectorUtils::compile_list_of_orders_for_server($ignore_delay, -1);
        
        if (empty($orders)) {
            WP_CLI::success('No orders found that need synchronization.');
            return;
        }

        WP_CLI::log(sprintf('Found %d orders to synchronize.', count($orders)));
        
        if ($dry_run) {
            WP_CLI::log("\nOrders that would be synchronized:");
            $count = 0;
            foreach ($orders as $order_id => $order_data) {
                $count++;
                $order = wc_get_order($order_id);
                WP_CLI::log(sprintf('- Order #%d (%s) - Member: %s', 
                    $order_id, 
                    $order ? $order->get_date_created()->format('Y-m-d H:i:s') : 'Unknown date',
                    $order ? $order->get_meta('dahannes_member_code', true) : 'No member code'
                ));
                if ($count >= 10 && count($orders) > 10) {
                    WP_CLI::log(sprintf('... and %d more orders', count($orders) - 10));
                    break;
                }
            }
            WP_CLI::success('Dry run completed. No orders were synchronized.');
            return;
        }

        // Process orders in batches
        WP_CLI::log(sprintf('Processing orders in batches of %d...', $batch_size));
        
        // Sort orders by ID (oldest first)
        ksort($orders);
        
        // Split orders into batches
        $batches = array_chunk($orders, $batch_size, true); // true preserves keys
        
        $success_count = 0;
        $failed_count = 0;
        
        // Create progress bar for batches
        $progress = \WP_CLI\Utils\make_progress_bar('Synchronizing order batches', count($batches));
        
        foreach ($batches as $batch_index => $batch) {
            try {
                $result = $client->send_order_list($batch);
                if ($result) {
                    $success_count += count($batch);
                    // Mark orders as submitted
                    $mark_result = DahannesConnectorUtils::mark_orders_processed(array_keys($batch), $info_data['data_version']);
                    
                    WP_CLI::debug(sprintf('Successfully processed batch %d with %d orders', 
                        $batch_index + 1, $mark_result['success']));
                } else {
                    $failed_count += count($batch);
                    WP_CLI::warning(sprintf('Failed to process batch %d with %d orders', 
                        $batch_index + 1, count($batch)));
                }
            } catch (\Exception $e) {
                $failed_count += count($batch);
                WP_CLI::warning(sprintf('Exception processing batch %d: %s', 
                    $batch_index + 1, $e->getMessage()));
            }
            
            $progress->tick();
        }
        
        $progress->finish();
        
        // Final results
        WP_CLI::log('');
        WP_CLI::log(sprintf('Synchronization completed:'));
        WP_CLI::log(sprintf('- Successfully synchronized: %d orders', $success_count));
        
        if ($failed_count > 0) {
            WP_CLI::log(sprintf('- Failed to synchronize: %d orders', $failed_count));
            WP_CLI::warning('Some orders failed to synchronize. Check logs for details.');
        } else {
            WP_CLI::success('All orders synchronized successfully!');
        }
    }

    /**
     * Compiles a list of orders for server synchronization without the 200 order limit.
     * This is a modified version of DahannesConnectorUtils::compile_list_of_orders_for_server()
     * 
     * @return array Array of orders ready for synchronization
     */
    private function compile_unlimited_orders_for_server(): array {
        // Get shop type information
        $info_data = get_option('dahannes_info', []);
        $shop_type = $info_data['app_type'] ?? '';
        $data_version = $info_data['data_version'] ?? '';
        
        // Get orders in two batches: tracked orders needing resubmission + untracked orders with member codes
        $all_orders = [];
        
        try {
            // NO LIMIT - process all orders that need synchronization
            
            // First: Get orders with submission_status < data_version (already tracked, need resubmission)
            $tracked_orders_args = [
                'limit' => -1, // No limit
                'meta_query' => [
                    [
                        'key' => DahannesConnectorUtils::DAHANNES_SUBMISSION_STATUS,
                        'value' => (int)$data_version,
                        'compare' => '<',
                        'type' => 'NUMERIC'
                    ]
                ],
                'type' => 'shop_order',
            ];
            $tracked_orders = wc_get_orders($tracked_orders_args);
            
            WP_CLI::debug(sprintf('Found %d tracked orders needing resubmission', count($tracked_orders)));
            
            // Second: Get orders WITHOUT submission_status (untracked orders) - no limit
            $untracked_orders_args = [
                'limit' => -1, // No limit
                'meta_query' => [
                    [
                        'key' => DahannesConnectorUtils::DAHANNES_SUBMISSION_STATUS,
                        'compare' => 'NOT EXISTS'
                    ]
                ],
                'type' => 'shop_order',
            ];
            $untracked_orders = wc_get_orders($untracked_orders_args);
            
            WP_CLI::debug(sprintf('Found %d untracked orders to check', count($untracked_orders)));
            
            // Combine both sets of orders
            $all_orders = array_merge($tracked_orders, $untracked_orders);
            
            // Remove duplicates based on order ID
            $unique_orders = [];
            foreach ($all_orders as $order) {
                $unique_orders[$order->get_id()] = $order;
            }
            $orders = array_values($unique_orders);
            
        } catch (\Exception $e) {
            WP_CLI::error('Error retrieving orders: ' . $e->getMessage());
            return [];
        }
        
        if (empty($orders)) {
            return [];
        }
        
        WP_CLI::debug(sprintf('Processing %d unique orders for sync eligibility', count($orders)));
        
        $orders_for_server = [];
        $processed = 0;
        $skipped = 0;
        $skipped_order_ids = []; // Collect orders to mark as processed in batch
        
        foreach ($orders as $order) {
            $order_id = $order->get_id();
            
            // Skip refunds - check the order type directly
            if ($order->get_type() !== 'shop_order') {
                $skipped_order_ids[] = $order_id;
                $skipped++;
                continue;
            }
            
            $code = DahannesConnectorUtils::get_member_code_from_order($order);
            
            // For partner shops: skip orders without a code or with invalid code length
            if ($shop_type === 'partner' && (empty($code) || strlen($code) < 5 || strlen($code) > 7)) {
                $skipped_order_ids[] = $order_id;
                $skipped++;
                continue;
            }
            
            // For non-partner shops, include orders even if code is empty
            $qualifies_for = $order->get_meta(DahannesConnectorUtils::DAHANNES_QUAL_SLUG, true);

            // Calculate subtotal minus discounts, refunds, and products with disabled bonus
            $order_total = $order->get_total();
            $percentage_refunded = $order_total > 0 ? $order->get_total_refunded() / $order_total : 0;

            // Calculate subtotal only for products that don't have bonus disabled
            $subtotal = 0;
            foreach ($order->get_items() as $item) {
                $product_id = $item->get_product_id();
                
                // Skip products with disabled bonus - HPOS compatible
                $product = wc_get_product($product_id);
                $disable_bonus = $product ? $product->get_meta('_dahannes_disable_bonus', true) : '';
                if ($disable_bonus === 'yes') {
                    continue;
                }
                
                // Add item subtotal to order subtotal
                $subtotal += $item->get_subtotal();
            }
            
            $subtotal_not_refunded = $subtotal * (1 - $percentage_refunded);

            // Determine if order is active (based on status only)
            $is_active = in_array($order->get_status(), ['completed', 'processing']);
            
            $orders_for_server[$order_id] = [
                'order_id' => $order_id,
                'order_date' => $order->get_date_created()->format('Y-m-d H:i:s'),
                'order_total' => $subtotal_not_refunded,
                'order_qualifies_for' => $qualifies_for,
                'member_code' => $code, // Keep code as is, even if empty
                'active' => $is_active,
            ];
            $processed++;
        }
        
        // Batch update skipped orders to mark them as processed
        if (!empty($skipped_order_ids)) {
            DahannesConnectorUtils::mark_orders_processed($skipped_order_ids, $data_version);
            
            WP_CLI::debug(sprintf('Marked %d skipped orders as processed', count($skipped_order_ids)));
        }
        
        WP_CLI::debug(sprintf('Order processing results: %d eligible, %d skipped', $processed, $skipped));
        
        return $orders_for_server;
    }

    /**
     * Tests the Sentry logging integration.
     *
     * ## OPTIONS
     *
     * [--level=<level>]
     * : Log level to test (debug, info, warning, error). Default: all levels.
     *
     * ## EXAMPLES
     *
     *     # Test all log levels
     *     $ wp dahannes test_sentry
     *
     *     # Test only error level
     *     $ wp dahannes test_sentry --level=error
     *
     * @when after_wp_load
     */
    public function test_sentry($args, $assoc_args) {
        $level = $assoc_args['level'] ?? 'all';

        WP_CLI::line('Testing Sentry logging integration...');
        WP_CLI::line('');

        // Show current Sentry configuration
        $sentry_enabled = defined('DAHANNES_ENABLE_SENTRY') && DAHANNES_ENABLE_SENTRY;
        WP_CLI::line('Sentry Status: ' . ($sentry_enabled ? 'ENABLED' : 'DISABLED'));
        
        if (!$sentry_enabled) {
            WP_CLI::error('Sentry is disabled. Set DAHANNES_ENABLE_SENTRY=true in wp-config.php');
            return;
        }

        $environment = DahannesLogger::get_current_environment();
        $context = DahannesLogger::get_current_installation_context();
        $installation_id = parse_url($context['site_url'], PHP_URL_HOST);

        WP_CLI::line('Environment: ' . $environment);
        WP_CLI::line('Installation ID: ' . $installation_id);
        WP_CLI::line('');

        $test_context = [
            'test_type' => 'wp_cli_test',
            'timestamp' => current_time('mysql'),
            'user' => 'wp-cli',
            'command' => 'wp dahannes test_sentry'
        ];

        $levels_to_test = [];
        
        if ($level === 'all') {
            $levels_to_test = ['debug', 'info', 'warning', 'error'];
        } else {
            $levels_to_test = [$level];
        }

        foreach ($levels_to_test as $log_level) {
            WP_CLI::line("Sending {$log_level} level log...");
            
            $message = "WP-CLI Sentry test: {$log_level} level message";
            $level_context = array_merge($test_context, ['severity' => $log_level]);
            
            DahannesLogger::log($log_level, $message, $level_context);
            
            WP_CLI::line("✓ {$log_level} log sent");
        }

        WP_CLI::line('');
        WP_CLI::success('Test logs have been sent to Sentry! Check your Sentry dashboard.');
        WP_CLI::line('Expected tags: installation=' . $installation_id . ', environment=' . $environment);
    }

    /**
     * Manages plugin migrations and version tracking.
     *
     * ## OPTIONS
     *
     * <action>
     * : The action to perform: run, reset, status
     *
     * ## EXAMPLES
     *
     *     # Check current migration status
     *     $ wp dahannes migration status
     *
     *     # Reset migration version (for testing)  
     *     $ wp dahannes migration reset
     *
     *     # Force run migrations
     *     $ wp dahannes migration run
     *
     * @param array $args Command arguments
     * @param array $assoc_args Associative arguments
     */
    public function migration($args, $assoc_args) {
        if (empty($args)) {
            WP_CLI::error('Please specify an action: status, reset, or run');
        }
        
        $action = $args[0];
        
        switch ($action) {
            case 'status':
                $current_version = \DaHannesConnector\PLUGIN_VERSION;
                $migration_version = DahannesMigrationManager::get_migration_version();
                
                WP_CLI::line("Current Plugin Version: {$current_version}");
                WP_CLI::line("Last Migration Version: {$migration_version}");
                
                if (version_compare($migration_version, $current_version, '<')) {
                    WP_CLI::warning('Migrations are needed');
                } else {
                    WP_CLI::success('All migrations up to date');
                }
                break;
                
            case 'reset':
                DahannesMigrationManager::reset_migration_version();
                WP_CLI::success('Migration version reset. Next plugin load will run all migrations.');
                break;
                
            case 'run':
                DahannesMigrationManager::maybe_run_migrations();
                WP_CLI::success('Migrations completed.');
                break;
                
            default:
                WP_CLI::error("Unknown action: {$action}. Use: status, reset, or run");
        }
    }
}

// Register the command
WP_CLI::add_command('dahannes', 'DaHannesConnector\includes\DahannesCLICommands');
